// ==UserScript==
// @name          ExtensionOptionsMenu.uc.js
// @description   Symbolleistenschaltfläche zur Add-ons-Verwaltung
// @include       main
// @charset       UTF-8
// @version       3.1.8 Ergänzungen und Anpassungen an Thunderbird 78 von bege
// @version       3.1.7 Einstellbarer Dateimanager (Z. 48,49), alle Einstellungsseiten können geöffnet werden
// @version       3.1.6 Anpassung an Firefox 65 von aborix
// @version       3.1.5 Deaktivieren der Erweiterungen mit Rechtsklick repariert von aborix
// @version       3.1.4 Anpassung an Firefox 61+ von aborix
// @version       3.1.2 Auf Grund von Problemen, wurde die Fensteranzeige der Erweiterungseinstellungen entfernt
// @version       3.1.1 Öffnen des Erweiterungen-Ordners funktionierte nicht
// @version       3.1.0 Unterstützt Fx 57 oder später. System Add-ons, konnten nicht deaktiviert werden
// @version       3.0.10 Fx52以降で右クリック時に出る既存メニューを出ないように修正 (2ch該当スレより拝借)
// @version       3.0.9 表示したくないアドオンを設定できるように(コメントアウト内の説明参照)、選択アドオンのidをAlt+左クリックでコピーできるように
// @version       3.0.8 再起動に関する修正
// @version       3.0.7 メニューにアイコンが出ていなかったのを修正
// @version       3.0.6 互換性を考慮して書き換え デフォルトでボタンはカスタマイズパレットに配置
// @version       3.0.5 ツールチップに操作法を表示するように コメントアウト内CSSを更新
// @version       3.0.4 一部アドオンの設定画面が機能していなかったのを修正、メニューパネル内でドロップマーカーが出ないようにするCSSを追記
// @version       3.0.3 ボタンをツールバーパレットから自由に配置できるように変更(メニューパネル内への配置にも対応 ※コメントアウト内のcssを追加するように)
// ==/UserScript==
/*

Schaltflächensymbol:
Linksklick: Erweiterungsliste anzeigen
Mittelklick: Thunderbird neustarten
Rechtsklick: Add-ons-Manager öffnen

Erweiterungen:
Linksklick: Erweiterungseinstellungen öffnen, wenn vorhanden
Shift+Linksklick: Einstellngs-URL kopieren
Mittelklick: Internetseite der Erweiterung öffnen
Rechtsklick: Erweiterungen aktivieren/deaktivieren
Strg + Linksklick: Erweiterungsordner öffnen
Strg + Mittelklick: Erweiterungs-ID in Zwischenablage kopieren
Strg + Rechtsklick: Erweiterung entfernen / Deinstallation rückgängig machen
*Nach Thunderbird-Neustart kann dies nicht mehr Rückgängig gemacht werden.

Es werden auch die Einstellungsseiten in einem neuen Tab geöffnet, die sonst innerhalb des Add-on-Managers öffnen.
Falls das nicht gewünscht ist, die Zeilen 395 - 397 auskommentieren.

Bei "blackListArray" ab Zeile 68 können Add-ons, die nicht im Menü angezeigt werden sollen,
mit der ID des Add-ons eingegeben werden. Sie können die ID mit Strg + Mittelklick kopieren.
Einige Systemerweiterungen werden standardmäßig hinzugefügt.

Dateimanager:
Die Erweiterungs-Ordner können mit einem eigenen Dateimanager geöffnet werden.
Dazu den Pfad in Zeile 59 und eventuell einen Parameter in Zeile 60 eintragen.
Falls der Standard-Dateimanager des Systems verwendet werden soll, diese Zeilen auskommentieren.

Sortierung: Zeilen 93 - 94

*/

'use strict';

window.EOM = {

	vFileManager: 'C:\\Program Files\\FreeCommander XE\\FreeCommander.exe',
	vFMParameter: '/T',
	showVersion:  true,    // Versionsinfo anzeigen (true = Versionsinfo anzeigen, false = nicht anzeigen)
	showAll:      true,    // Alles anzeigen, auch bei Erweiterungen ohne Einstellungen
	showDisabled: true,    // deaktivierte Erweiterungen anzeigen (deaktivierte Add-ons anzeigen)
	autoRestart:  false,   // Firefox nach Installation bzw. Deinstallation automatisch neu starten
	iconURL:      'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAAAZdEVYdFNvZnR3YXJlAFBhaW50Lk5FVCB2My41LjbQg61aAAACkUlEQVQ4T43T60tTYRwH8HMQ9QQJRSBJ50xU8BL1QpJMsbxNc162edxcYlAoZdkFh6gZurF5WV6nc7M/oBdBb7q9DSPEVBDbZtN0c5tzNymolwXht2eDhVO0Dnx4Hn6/5/me8xx4KOqQR2rcYfjpIC81BpXiqWBnxUSgpWQ0kHrY+gN1xdOdu/XTQfDGIMSGAET6AMpG/TbhiD/uv0LqTYF7cmPgN2/wQzzhh2jMB+Gwz1I65I3/Z8A1o5eRTXqP85M+pVTv260Z86JieNtcMridXNjnZvI1Lia31xV7IIgf99AKg/e1wrAN+YQHtXoPJKNbqBrewlWdG6UDLlzRupCv3sTFns3vFx47SqJCFHoPoyAb5eNb4MlGyYgb1UNuiHQulPW7UKRx4rJqE5d6HMjpdiC7066mRFpHvFTnbCHuSJ84E+rIJumQExKdEzVE5YAT5RoHCnvsyO3aQHb7Os63rSHrwRoy76+qqErNBi/ut4PYrdFsKCWDDoj77CjvXUdu+yqyWleQcsuK5GYrBE0WcE0Wm6DZmsk1W7VEI1XRu6YUqb6gUh22W9BhQ8ZtCwQ3PoEjQuM+psi5SSBNCR/Zusq7bSju+IyMpmWwjUvgrh+hcWks6scVKs0tBQ/NSG5YBKtYNHOKRRxt4WUogKufTwmh8lqXU9MaFlY42UcLJ5tnOfk8yPwov0j/LfGNUIe/huXnYrm6uTiOn2UI7GEjcxMxTrwifu7rq6KOw0o+MAT2SI8sYGtnaVJ/s68fFUCfONd2jK2e+cFWv0dY1bu+mPiTocsTmyR8kU56X//2wmtmuiMvoMkkdEkEp3K0N08XPZsKScwzdNB0zFlSz0pIaxBG6mQ0JBU/1yXmm878AbFQoHrb98HyAAAAAElFTkSuQmCC',

	blackListArray: [
		"clicktoplay-rollout@mozilla.org",
		"firefox@getpocket.com",
		"screenshots@mozilla.org",
		"followonsearch@mozilla.com",
		"formautofill@mozilla.org",
		"onboarding@mozilla.org",
		"shield-recipe-client@mozilla.org",
		"webcompat@mozilla.org",
		"activity-stream@mozilla.org",
		"presentation@mozilla.org",
		"aushelper@mozilla.org",
		"webcompat-reporter@mozilla.org",
		"e10srollout@mozilla.org",
		"google@search.mozilla.org",
		"ecosia@search.mozilla.org",
		"ebay@search.mozilla.org",
		"doh-rollout@mozilla.org",
		"bing@search.mozilla.org",
		"amazon@search.mozilla.org",
		"ddg@search.mozilla.org",
		"leo_ende_de@search.mozilla.org",
		"wikipedia@search.mozilla.org"
	],

	sort: {
		enabled: 0,
		disabled: 1
			// 0, 0 - In alphabetischer Reihenfolge
			// 0, 1 - Reihenfolge wie im Add-On-Manager
	},

	init: function() {

		var style = `
			@namespace url('http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul');
/* 			#eom-button {
				list-style-image: url('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAAAZdEVYdFNvZnR3YXJlAFBhaW50Lk5FVCB2My41LjbQg61aAAACkUlEQVQ4T43T60tTYRwH8HMQ9QQJRSBJ50xU8BL1QpJMsbxNc162edxcYlAoZdkFh6gZurF5WV6nc7M/oBdBb7q9DSPEVBDbZtN0c5tzNymolwXht2eDhVO0Dnx4Hn6/5/me8xx4KOqQR2rcYfjpIC81BpXiqWBnxUSgpWQ0kHrY+gN1xdOdu/XTQfDGIMSGAET6AMpG/TbhiD/uv0LqTYF7cmPgN2/wQzzhh2jMB+Gwz1I65I3/Z8A1o5eRTXqP85M+pVTv260Z86JieNtcMridXNjnZvI1Lia31xV7IIgf99AKg/e1wrAN+YQHtXoPJKNbqBrewlWdG6UDLlzRupCv3sTFns3vFx47SqJCFHoPoyAb5eNb4MlGyYgb1UNuiHQulPW7UKRx4rJqE5d6HMjpdiC7066mRFpHvFTnbCHuSJ84E+rIJumQExKdEzVE5YAT5RoHCnvsyO3aQHb7Os63rSHrwRoy76+qqErNBi/ut4PYrdFsKCWDDoj77CjvXUdu+yqyWleQcsuK5GYrBE0WcE0Wm6DZmsk1W7VEI1XRu6YUqb6gUh22W9BhQ8ZtCwQ3PoEjQuM+psi5SSBNCR/Zusq7bSju+IyMpmWwjUvgrh+hcWks6scVKs0tBQ/NSG5YBKtYNHOKRRxt4WUogKufTwmh8lqXU9MaFlY42UcLJ5tnOfk8yPwov0j/LfGNUIe/huXnYrm6uTiOn2UI7GEjcxMxTrwifu7rq6KOw0o+MAT2SI8sYGtnaVJ/s68fFUCfONd2jK2e+cFWv0dY1bu+mPiTocsTmyR8kU56X//2wmtmuiMvoMkkdEkEp3K0N08XPZsKScwzdNB0zFlSz0pIaxBG6mQ0JBU/1yXmm878AbFQoHrb98HyAAAAAElFTkSuQmCC');
			}
 */
			.addon-disabled > .menu-iconic-left { filter: grayscale(1); }
			.addon-disabled label { color: Gray !important; }
			.addon-uninstall label { font-weight: bold !important; }
			.addon-uninstall label:after { content: '-'; }

			// #eom-button[cui-areatype="menu-panel"],
			// toolbarpaletteitem[place="palette"] > #eom-button {
				// list-style-image: url('data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAFHklEQVRYw8WXXYhd1RXHf+vcOxPj+JGYBow6M2rb1FISI77EkEDEB1GiEkdN24ciCYXSmtI+BBJBBdtEiYhCDCgkioIPmsSvByUY6BQ1pgi2Tj+gLU2aGUJJp5PpfNzcmXv3Xn8f9rlfmdwZkgm4YLEP+5y99n+v9V9r7WOS+CaleDGLNu79b29mehTjXonFEosEHdGpRteoi9Mh8l5w3uzfdu3p89m4bcdx/vjMzRcOoG/v6awjs5fM2CCBk1QSBhgszWB5wbTOjeV3Pvefn/9u2zJvZy+7UAALCra4YNyCoJ0qHwvwnaKx5BKHQFZfJ+WbKk1LjTkA0TGXtQv2QKXqkyH4V8hLyo8qCeT5mFTOeIw+UA0+cUkBHNy6bCpE/SxG/UquL+USDnKBK+3tOurRt4aoJ/t33DB1SULwo72nFgGGo05j7K2t1+978MVTtwKr5BhSLfZRUR8f2d79xl27Brvu3HlyUXRwCRd4JHMxBkQAm6sO/HDPqbuB7cD6mrsliAJhuDukw+cq3Mm/EdHB3RBOjAmERH8Uz37x25sPz+qBvheGso5idqk3J4r17gKYHUDR6ALW455zKzeS571yxje4J+QNMKlICJdQ4kcdrFzr5uSAQRF3FIVonKCWedTi2gQAb31ufEwqVJ5GicIMAH27T3wLoBp8/IPHv12RHNyIngA0ZRlOKwAElqPLcg9FgTt47gV3EWPC5K4GCe/9zb+6O4q2p5DZ9wFi1J+j9NjCThuTFc5GjyjtWI85ZvVnCaKLStWpBNU1OgQXMaoOuh42iCde+UGxmFyjzZl4wPMAG9yYuX94YNt3X924+zgKyYDRcGWIolJxShWnXBFTFacaEwld6dTJ1wCcMGOX4G/AL4AfQ3MI3O+JWDKehk4Tj93/9D9WZcUCeGo0LnF2OjJRFpPlyFTVCVFoJneStcaLl7HstcF9K2LvloEB4H1gaR2AXItjDhjLR1gFrPQQVY2yibOR/5cipelYj6PlgG32GnYGs4GT+1ZEgJP7V072bhk4UKvCNQBLlPvecjdkGRaqKoyVIiOTgXLFcUFmTSdsEHw2mQSVmidO7l+p3s1fxToAd3VYk/+EmJiMDI8HJsspl7LMUtAu/AJVBcKMnloqN3pBCD6dAZYlNo+OR0YmqoSYNjYjlb+Lkw5QcdZm5CGOWGZLQ4Dh8Sqj4wEDChkBV4E5wzyrLEK6os3eIQGI/j+5MTJWZXQiYIkDnyGOYPZUSvyLlqtAq4GPAHo2HbsO2AgsAZ7OQ6D3y9Nh7ehEIEt1IeI8P/j26nd7Nv3hKeZ/c/5pzyPHpoF/Aj8BNuTzCUA16PXxydBL9FusADh/Aft9z0NHu8iyVE/nJ8uAnefMzbwPdPcdXZoXotHBg2tCT9+ni7HsTF6H2+c5TOQGi8DVuc4lcfDQ2mILO4cOrRluSRVJRn7XmynHwfab8SVQagJwpeAOpEeB6+f3Y+JCGbQh4c6h99a9er4X3Q9+ehhXCdg1v1uxWpr9ufrXdsuG3lkbkP7eZl1D5wTgylquMa36y+77+s8b6+77+rtxbW6zrqaF8wKwJAUz6xz+/NcVYvikzS/Qw14Zu7/zqpuuMLPLa1q87JouD6WHQHcz2+8TGjIzK567eQ7qMuDK6ZE/dU0Nf/HGgmtWLLBs4e35u1pKdMTy8Pfi1MiNtZRKjS0UvHzmpqxrYbF5vkkKyP/tofQEkLWk4bkAgK5Uy+dViltYlTens8A4MPU1jYiz7Re8VU8AAAAASUVORK5CYII=');
			// }
		`;

		style = style.replace(/\s+/g, " ");
		var sspi = document.createProcessingInstruction(
			'xml-stylesheet',
			'type="text/css" href="data:text/css,' + encodeURIComponent(style) + '"'
		);
		document.insertBefore(sspi, document.documentElement);
		sspi.getAttribute = function(name) {
			return document.documentElement.getAttribute(name);
		};

		const XUL_NS = 'http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul';
		const eombutton = $C("menu", {
			id: "eom-button",
			label: "Add-ons",
			tooltiptext: "Extension Options Menu\n\nLinksklick: Erweiterungsliste anzeigen\nMittelklick: Thunderbird neu starten\nRechtsklick: Add-ons-Manager öffnen",
			className: "menu-iconic",
			accesskey: "O"
		});

		$('mail-menubar').appendChild(eombutton);

		var btn = document.getElementById('eom-button');
		btn.addEventListener('click', EOM.iconClick);

		var mp = btn.appendChild(document.createElementNS(XUL_NS, 'menupopup'));
		mp.setAttribute('id', 'eom-button-popup');
		mp.setAttribute('onclick', 'event.preventDefault(); event.stopPropagation();');
		mp.addEventListener('popupshowing', (event) => EOM.populateMenu(event));

		function copyList() {
			AddonManager.getAddonsByTypes(["extension"]).then(function(extensions) {
				Cc['@mozilla.org/widget/clipboardhelper;1'].getService(Ci.nsIClipboardHelper).copyString(extensions.map(function(item, id) {
					return (id + 1 + ". " + item.name + " [" + item.version + "]" + "\nID:" + item.id + "\naktiv:" + item.isActive + "\n----------");
				}).join("\n"));
			});

			var alertsService = Cc['@mozilla.org/alerts-service;1'].getService(Ci.nsIAlertsService);
			alertsService.showAlertNotification("data:image/png;base64,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", "Add-ons-Liste", "Add-ons-Liste\nin die Zwischenablage kopiert.", null, null, null, "copyList");
			setTimeout(function() {
				alertsService.closeAlert("copyList");
			}, 5000);
		};

		var mMenus = [
			{
				alabel: 'Thunderbird neu starten',
				label: 'Script-Cache löschen',
				image: "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAACgElEQVQ4jY2RfUzMARjHv7tODnmJOxGm3LnKe3fnoh+W184ypjmZpZrQFLOSstns5g/cIXq9fuqQUd4tx0jFcLVRrSxNNE2bsUYY5Sr09Y9u2Nz6/vk83+ez5/s8gBvFAbKCUKw7Hz6o3KrDDHfev5Qmx/BCAVvKklR1b8rSWHMovM+ignJAw6IeEZU7FC3tNxeSjWvJF8l8Z0/tu5eyqKloiWd6MjDELcCqg/5hqk8bm8LIulCyQiCrjGRVCjuupbN04+Tygyoo3EIypkNVluDd0OsIJe+F8KV5IjtFFXkhnM7iRF5eM+aaEfBwDeTpEGDVQcgLwTyTAl4AIGqhrNg+uvlzaTBti3D0nEGa2W6ZRNoW87VpAfPnwuAC2I1eLa3FMT8cphVOUQtNfz1XA1XJqkH3bQJWAkBJhMcZ54mp/Hl4Fq8aPM+5AFUxsi42JLFR3PwtQ40J/ySShAHS31sFPt873smjKjqihr5yOSo3DH7NO2vZkm/8njUb+v/dJg6Q1e6Sv2FOIOs3jfzqalxYjlM/CrXsvrWVxSs9TwFAjh7q0wKsohbyft8RJcZWJ4zp+nTAj4/WD/v45+vCWtN9SHsk2zINLJiPvVYdNjRbo2mP9X9i8cM4ADAp4FUoINYmIP6kgNV/5bwaIS3tOaEmr0Tybe5qPtg553N3dRa/1Yi8ETvNYQ6A7/+iAQDMAfC9bZQ97jT7k0ULyevR5KUo8qzAnrt7WJ6oeSpqMdMtRNRCXrJMkl27bWTHh/3jfzJDSWb4s/eYmg37QliwALvdAvplCcJUR8yI953mKayP9/5ycRls2cHQAZAMCGDyw6grBumz4qUS83ENgtx5fwEzyhRmLMK7zwAAAABJRU5ErkJggg==",
				oncommand: "Services.appinfo.invalidateCachesOnRestart() || BrowserUtils.restartApplication();",
				style: "min-width: 260px;"
			},
			{
				label: "Erweiterungsordner öffnen",
				image: "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAADEElEQVQ4jY2RXUiTcRjF/9JFpBAapk5NnWzzI5vTbb5uuunrtFm5XCpi7tVEs0zU9aFFk6wmiV9RfrRqkRgZhVIZSjXUNDPCmaamlZQQ0gdFVxGFbu10paXzonN5Hs6P5zkPISsVniYjArXAzv8vceVyIi8A71g7hNW9k56eQsfFEYeQtUlOzqFJ69dzV4uuIbw4LxLB7CCyfNDGccgujcE9rqgvM4D6ZAjmvKjm+HYUbWShLYxn65Rsfro87iHwI9H5YBUYsankGqQXnkNycQyBlSaIK+7i6x4pblFBn/e6usMUswVP4vgzjKMr6y/ANYhFonIR1WxGTMsrSI2TEBnGwG8cgUjfjY+7JeiL5eM8zx/jieEYUYThPhVireP6Zi4iHEhk9im/Q20vvAuvQNBoRkjDMJry9mM0NRrv0yi8U0fgTZIIU4lCjNECm1kuQDXbh/m7RVzxARJ/pJLI8uF3oguc+iG0ZqSiR03jbbIYw2oRLhdSMCvCYIoIfqZycfH5twUHIs1d2LDXgI3F1+Bf8xjeVf1w1/fAu/QmprcJUX9UCk27EvcSQtEZHjRo94Z18qwPXsc64FczCK8zj+B2+iHoWiNS9BVo04hwSB+FlNZ45FRIoaigPtgBjuZtvlXZUIDx4cNIb2rGhvJOfDFrYOpVIePmVqS0JkBlVEDZSEN8Ujy7FExRurIMx0N0tdrA0S5jPKxzJdA0n4OHrg1fzAxeDqpxp0sJ7VUaygYa7JKA64SQNUuAg7t9yw06PoY7d+F1vwbWuRL8nNmHH1M5sEwzmJ9Ih2VUDX1LLGJrYsDRhsAjj3t7CcAkuYW2N9LfrF91sH4qg3VOC8tsAb5PZMMyzWDApMLOszLIqmQ2ySkZhMejEFAknFx2/8EsbtCD1sSpoY5kWOe0MF2NHzhTxPv9a1KD+907EK4T2/ilIoSWRdrc0tmMk8Rli12JRzTstK4rCfML74ttN+qo5NIstqq3ha46fThY4Ug7J7MY7rfgYspCBM7OduFFZW/34uWm+vivOgxw9HSiXPgr7T+DX3N5gyCN2AAAAABJRU5ErkJggg==",
				oncommand: "EOM.browseExtDir();"
			},
			{
				label: "Add-ons-Liste in die Zwischenablage kopieren",
				image: "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAABlSURBVDhP5Y5BCsAgEAP3i/1AP+D/zxUlwWBXXQueOhAQzQStcN3p2UmVFK80C7QGH1aEBniOBPqhgRnsQB8P8KzRe+i/+YHCO+htQNPjdaB/G4D6hoWekFzQohfUxngSg4pglgGUsQ0ZR4jGSwAAAABJRU5ErkJggg==",
				oncommand: '(' + copyList.toString() + ')()'
			}
		];

		var menugroup = mp.appendChild($C("menugroup", {
			id: "eom-menugroup"
		}));
		mp.appendChild($C('menuseparator', {
			id: "eom-separator1"
		}));

		for (let i = 0, menu; menu = mMenus[i]; i++) {
			menugroup.appendChild($C("menuitem", {
				label: menu.alabel,
				tooltiptext: menu.label,
				image: menu.image,
				class: "menuitem-iconic",
				oncommand: menu.oncommand,
				style: menu.style || "max-width: 10px;"
			}));
		};

		function $C(name, attr) {
			var el = document.createXULElement(name);
			if (attr) {
				Object.keys(attr).forEach(function(n) {
					el.setAttribute(n, attr[n]);
				});
			};
			return el;
		};
		
	},

	populateMenu: function(event) {
		var prevState;
		var showItem = true;

		var popup = event.target;
		if (popup !== event.currentTarget) {
			return;
		}

		while (popup.childElementCount > 2) {
			popup.removeChild(popup.lastChild);
		}

		var addons;
		AddonManager.getAddonsByTypes(['extension']).then(function(aAddons) {
			addons = aAddons;
		});

		var thread = Services.tm.mainThread;
		while (addons === void 0) {
			thread.processNextEvent(true);
		}

		addons.sort((a, b) => {
			var ka = this.key(a);
			var kb = this.key(b);
			return (ka < kb) ? -1 : 1;
		}).forEach((addon) => {
			if (!addon.appDisabled && ((addon.isActive && addon.optionsURL)
				 || ((addon.userDisabled && this.showDisabled)
				 || (!addon.userDisabled && this.showAll)))) {
				var state = addon.isActive;
				if (this.sort.disabled === 1 && (prevState && state !== prevState)) {
					popup.appendChild(document.createXULElement('menuseparator'));
				}
				prevState = state;

				var mi = document.createXULElement('menuitem');
				var label = addon.name;
				if (this.showVersion) label = label += ' ' + '[' + addon.version + ']';
				mi.setAttribute('label', label);
				mi.setAttribute('class', 'menuitem-iconic');
				mi.setAttribute('tooltiptext', 'ID: ' + addon.id + '\nEinstellungen: ' + addon.optionsURL + ' (Typ ' + Number(addon.optionsType) + ')\nHomepage: ' + addon.homepageURL + '\n\nLinksklick = Add-on-Einstellungen öffnen\nMittelklick = Add-on-Homepage öffnen\nRechtsklick = Add-on aktivieren / deaktivieren\nShift+Linksklick = Einstellungs-URL kopieren\nStrg + Linksklick = Add-on-Ordner öffnen\nStrg + Mittelklick = Add-on ID kopieren\nStrg + Rechtsklick = Add-on deinstallieren *\n* Diese Aktion kann nicht rückgängig gemacht werden.');
				// '\nGröße: ' + Math.floor(addon.size / 1024) + ' KB'		//funktioniert nicht
				var icon = addon.iconURL || addon.iconURL64 || this.iconURL || '';
				mi.setAttribute('image', icon);
				mi.addEventListener('click', (event) => this.handleClick(event));
				mi._Addon = addon;

				if (!addon.optionsURL && addon.isActive) {
					mi.setAttribute('style', 'font-style: italic');
				} else {
					if (!addon.operationsRequiringRestart) {
						mi.setAttribute('style', 'color: Blue');
					}
				}

				this.setDisable(mi, addon.userDisabled);
				this.setUninstall(mi, this.isPending(addon));

				if (this.blackListArray) {
					for (var i = 0; i < this.blackListArray.length; i++) {
						if (this.blackListArray[i] == addon.id.toLowerCase()) {
							showItem = false;
							break;
						} else {
							showItem = true;
						}
					}
				}
				if (showItem) popup.appendChild(mi);
			}
		});
	},

	iconClick: function(event) {
		if (event.target !== event.currentTarget) {
			return;
		}
		if (event.button === 1) {
			EOM.restart();
		} else if (event.button === 2) {
			// BrowserOpenAddonsMgr('addons://list/extension');
			openAddonsMgr();
		}
	},

	handleClick: function(event) {
		var mi = event.target;
		if (mi !== event.currentTarget) {
			return;
		}
		if (!('_Addon' in mi)) {
			return;
		}

		var addon = mi._Addon;
		var pending = this.isPending(addon);
		var hasMdf = event.ctrlKey || event.shiftKey || event.altKey || event.metaKey;

		switch (event.button) {
			case 0:
				if (!addon.optionsURL && addon.isActive && !hasMdf) {
				// window.alert('"' + addon.name + '" hat keine Einstellungen.');
				var alertsService = Cc['@mozilla.org/alerts-service;1'].getService(Ci.nsIAlertsService);
				var icon = addon.iconURL || addon.iconURL64 || this.iconURL || '';
				alertsService.showAlertNotification(icon, '"' + addon.name + '"', ' HAT KEINE EINSTELLUNGEN.', null, null, null, "noOptions");
				function beep() {
				var snd = new Audio("data:audio/wav;base64,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");  
					snd.play();
					}
					beep();
				setTimeout(function() {
				alertsService.closeAlert("noOptions");
				}, 8000);
			
				} else if (addon.optionsURL && !hasMdf && Number(addon.optionsType)) {
					this.openAddonOptions(addon);
				} else if (event.ctrlKey) {
					this.browseDir(addon);
				} else if (event.shiftKey) {
					Cc['@mozilla.org/widget/clipboardhelper;1'].getService(Ci.nsIClipboardHelper).copyString(addon.optionsURL);
				}
				break;
			case 1:
				if (addon.homepageURL && !hasMdf) {
					this.openDefaultBrowser(addon.homepageURL);
				} else if (event.ctrlKey) {
					Cc['@mozilla.org/widget/clipboardhelper;1'].getService(Ci.nsIClipboardHelper).copyString(addon.id);
				}
				break;
			case 2:
				if (!hasMdf) {
					var stateDisabled = !addon.userDisabled;
					stateDisabled ? addon.disable() : addon.enable();
					this.setDisable(mi, stateDisabled);
					if (addon.operationsRequiringRestart && this.autoRestart) {
						EOM.restart();
					}
				} else if (event.ctrlKey) {
					if (pending) {
						addon.cancelUninstall();
					} else {
						addon.uninstall();
					}
					this.setUninstall(mi, pending);
				}
				break;
		}
	},

	setDisable: function(mi, dis) {
		var cls = mi.classList;
		dis ? cls.add('addon-disabled') : cls.remove('addon-disabled');
	},

	setUninstall: function(mi, uninst) {
		var cls = mi.classList;
		uninst ? cls.add('addon-uninstall') : cls.remove('addon-uninstall');
	},

	isPending: function(addon) {
		return addon.pendingOperations & AddonManager.PENDING_UNINSTALL;
	},

	openAddonOptions: function(addon) {
		var optionsURL = addon.optionsURL || '';
		if (!addon.isActive || !optionsURL) {
			return;
		}
		switch (Number(addon.optionsType)) {
			case 3:
				"switchToTabHavingURI" in window ? switchToTabHavingURI(optionsURL, true) : openTab("contentTab", { contentPage: optionsURL });
			break;
			
	// auch nicht eigenständige Einstellungsseiten in neuem Tab öffnen ...
			case 5:
				"switchToTabHavingURI" in window ? switchToTabHavingURI(optionsURL, true) : openTab("contentTab", { contentPage: optionsURL });
			break;
	// ... falls das nicht gewünscht ist, zwischen diesen Kommentaren auskommentieren.
	
			default:
				openDialog(optionsURL, addon.name, 'chrome,titlebar,toolbar,resizable,scrollbars,centerscreen,dialog=no,modal=no');
		}
	},

	browseExtDir: function() {
		var extDir = Services.dirsvc.get('ProfD', Ci.nsIFile);
		var nsLocalDir = Components.Constructor('@mozilla.org/file/local;1', 'nsIFile', 'initWithPath');
		extDir.append('extensions');
			if (this.vFileManager.length != 0) {
				var file = Cc['@mozilla.org/file/local;1'].createInstance(Ci.nsIFile);
				var process = Cc['@mozilla.org/process/util;1'].createInstance(Ci.nsIProcess);
				var args=[this.vFMParameter,extDir.path];
				file.initWithPath(this.vFileManager);
				process.init(file);
				// Verzeichnis mit anderem Dateimanager öffnen
				process.run(false, args, args.length);
				} else {
				// Verzeichnis mit Dateimanager des Systems öffnen
				new nsLocalDir(extDir).reveal();
				}
	},

	browseDir: function(addon) {
		var dir = Services.dirsvc.get('ProfD', Ci.nsIFile);
		var nsLocalFile = Components.Constructor('@mozilla.org/file/local;1', 'nsIFile', 'initWithPath');
		dir.append('extensions');
		dir.append(addon.id);
		var fileOrDir = dir.path + (dir.exists() ? '' : '.xpi');
		try {
			if (this.vFileManager.length != 0) {
				var file = Cc['@mozilla.org/file/local;1'].createInstance(Ci.nsIFile);
				var process = Cc['@mozilla.org/process/util;1'].createInstance(Ci.nsIProcess);
				var args=[this.vFMParameter,fileOrDir];
				file.initWithPath(this.vFileManager);
				process.init(file);
				// Verzeichnis mit anderem Dateimanager öffnen
				process.run(false, args, args.length);
				} else {
				// Verzeichnis mit Dateimanager des Systems öffnen
				new nsLocalFile(fileOrDir).reveal();
				}
		} catch (e) {
			var addonDir = /.xpi$/.test(fileOrDir) ? dir.parent : dir;
			try {
				if (addonDir.exists()) {
					if (this.vFileManager.length != 0) {
					var file = Cc['@mozilla.org/file/local;1'].createInstance(Ci.nsIFile);
					var process = Cc['@mozilla.org/process/util;1'].createInstance(Ci.nsIProcess);
					var args=[this.vFMParameter,addonDir];
					file.initWithPath(this.vFileManager);
					process.init(file);
					// Verzeichnis mit anderem Dateimanager öffnen
					process.run(false, args, args.length);
					} else {
					// Verzeichnis mit Dateimanager des Systems öffnen
					addonDir.launch();
					}
				}
			} catch (e) {
				var uri = Services.io.newFileURI(addonDir);
				var protSvc = Cc['@mozilla.org/uriloader/external-protocol-service;1'].getService(Ci.nsIExternalProtocolService);
				protSvc.loadUrl(uri);
			}
		}
	},

	key: function(addon) {
		var sort = this.sort;
		var sortPos = addon.isActive ? sort.enabled : sort.disabled;
		return sortPos + '\n' + addon.name.toLowerCase();
	},
	
	openDefaultBrowser: function(url) {
          let uri = null;
          try {
            uri = Services.io.newURI(url);
          } catch (e) {
            throw new ExtensionError(`Url "${url}" seems to be malformed.`);
          }
          if (!uri.schemeIs("http") && !uri.schemeIs("https")) {
            throw new ExtensionError(
              `Url scheme "${uri.scheme}" is not supported.`
            );
          }
          Cc["@mozilla.org/uriloader/external-protocol-service;1"]
            .getService(Ci.nsIExternalProtocolService)
            .loadURI(uri);
        },

	// restart: Services.appinfo.invalidateCachesOnRestart && BrowserUtils.restartApplication
	restart: () => {
  Services.appinfo.invalidateCachesOnRestart();
  Services.startup.quit(Ci.nsIAppStartup.eRestart | Ci.nsIAppStartup.eAttemptQuit);
}

};
	var browser_chrome = 'chrome://messenger/content/messenger.xhtml';
	if (location == browser_chrome)
	EOM.init();

function $(id) { return document.getElementById(id); }
function $C(name, attr) {
	var el = document.createXULElement(name);
	if (attr) Object.keys(attr).forEach(function(n) { el.setAttribute(n, attr[n]) });
	return el;
}

