/* -*- Mode: javascript; tab-width: 20; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Calendar code.
 *
 * The Initial Developer of the Original Code is
 *   Joey Minta <jminta@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Matthew Willis <mattwillis@gmail.com>
 *   Ferdinand Grassmann <ferdinand@grassmann.info>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

var { Services } = ChromeUtils.import("resource://gre/modules/Services.jsm");
var { cal } = ChromeUtils.import("resource://calendar/modules/calUtils.jsm");
 
Components.utils.import("resource://calendar/modules/calUtils.jsm");
//Components.utils.import("resource://fgprinters/modules/calXMLUtils.jsm");
//Components.utils.import("resource://fgprinters/modules/calPrintUtils.jsm");
Components.utils.import("resource://fgprinters/modules/calHtmlUtils.jsm");
 
/**
 * Prints a rough month-grid of events/tasks
 */

function FGcalMonthPrinter() {
}

var FGcalMonthPrinterID = Components.ID('{E4F85D1C-CCBD-11DB-8915-EDC855D89593}');
var FGcalMonthPrinterInterfaces = [Components.interfaces.calIPrintFormatter];
var FGcalMonthPrinterContractID = "@mozilla.org/calendar/printformatter;1?type=fgmonthgrid";
var FGcalMonthPrinterDescription = "Calendar Month Print Formatter";

FGcalMonthPrinter.prototype = {
	classID: FGcalMonthPrinterID,
	
	QueryInterface: ChromeUtils.generateQI([Ci.calIPrintFormatter]),
	
	
	
	get name() {
        return "FG " + cal.l10n.getCalString("monthPrinterName"); 
    },

	formatToHtml : function FGmonthPrint_format(aStream, aStart, aEnd, aCount, aItems, aTitle) {

		// Parse the template file
		let html = cal.xml.parseFile("chrome://fgprinters/skin/printing/empty.html");
		let body = html.getElementsByTagName('body')[0];
		
		// Set page title
		html.getElementsByTagName("title")[0].textContent = aTitle;
	
		// Set the style sheet
		this.setupStyleSheet(html);

		// Title and Subtitle
		let printtitle = Services.prefs.getBoolPref("fgprinters.printtitle", false);
		let subTitle = Services.prefs.getCharPref("fgprinters.subtitle", "");
		if (printtitle && aTitle) {
			let p_aTitle = html.createElementNS("http://www.w3.org/1999/xhtml","p");
			p_aTitle.className = 'main-title';
			p_aTitle.textContent = aTitle;
			body.appendChild(p_aTitle);
		}
		if (subTitle) {
			let p_subTitle = html.createElementNS("http://www.w3.org/1999/xhtml","p");
			p_subTitle.className = 'sub-title';
			p_subTitle.textContent = subTitle;
			body.appendChild(p_subTitle);
		}		
	
		// If aStart or aEnd weren't passed in, we need to calculate them based on
		// aItems data.
	
		let start = aStart;
		let end = aEnd;
		if (!start || !end) {
			for (let item of aItems) {
				let itemStart = item.startDate || item.entryDate;
				let itemEnd = item.endDate || item.dueDate;
				if (!start || (itemStart && start.compare(itemStart) == 1)) {
					start = itemStart;
				}
				if (!end || (itemEnd && end.compare(itemEnd) == -1)) {
					end = itemEnd;
				}
			}
		}
		// Play around with aStart and aEnd to determine the minimal number of
		// months we can show to still technically meet their requirements.  This
		// is most useful when someone printed 'Current View' in the month view. If
		// we take the aStart and aEnd literally, we'll print 3 months (because of
		// the extra days at the start/end), but we should avoid that.
		//
		// Basically, we check whether aStart falls in the same week as the start
		// of a month (ie aStart  is Jan 29, which often is in the same week as
		// Feb 1), and similarly whether aEnd falls in the same week as the end of
		// a month.
		let weekStart = Services.prefs.getIntPref("calendar.week.start", 0);
		maybeNewStart = start.clone();
		maybeNewStart.day = 1;
		maybeNewStart.month = start.month+1;
		try {
			maybeNewStart.normalize();
		} catch(ex) {}
	
		let date = start.clone();
	
		// First we have to adjust the end date for comparison, as the
		// provided end date is exclusive, i.e. will not be displayed.
	
		let realEnd = end.clone();
		realEnd.day -= 1;
		try {
			realEnd.normalize();
		} catch(ex) {}
	
		if (start.compare(realEnd) <= 0) {
			// Only adjust dates if start date is earlier than end date.
	
			if ((start.month != realEnd.month) || (start.year != realEnd.year)) {
				// We only need to adjust if start and end are in different months.
	
				// We want to check whether or not the start day is in the same
				// week as the beginning of the next month. To do this, we take
				// the start date, add seven days and subtract the "day of week"
				// value (which has to be corrected in case we do not start on
				// Sunday).
				let testBegin = start.clone();
				let startWeekday = testBegin.weekday;
				if (startWeekday < weekStart) {
					startWeekday += 7;
				}
				testBegin.day += 7 + weekStart - startWeekday;
				try {
					testBegin.normalize();
				} catch(ex) {}
				if (testBegin.compare(maybeNewStart) > 0) {
					start = maybeNewStart;
					date = start.clone();
				}
			}
			if ((start.month != realEnd.month) || (start.year != realEnd.year)) {
				// We only need to adjust if start and end are in different months.
	
				// Next, we want to check whether or not the end day is in the same
				// week as the end of the previous month. So we have to get the
				// "day of week" value for the end of the previous month, adjust it
				// if necessary (when start of week is not Sunday) and check if the
				// end day is in the same week.
	
				let lastDayOfPreviousMonth = end.clone();
				lastDayOfPreviousMonth.day = 0;
				try {
					lastDayOfPreviousMonth.normalize();
				} catch(ex) {}
				let lastDayWeekday = lastDayOfPreviousMonth.weekday;
				if (lastDayWeekday < weekStart) {
					lastDayWeekday += 7;
				}
				if (date.month != end.month) {
					date.day = 1;
				}
				if ((lastDayWeekday + end.day - 1) < (7 + weekStart)) {
					date.day = end.day;
				}
				try {
					date.normalize();
				} catch(ex) {}
	
				// Finally, we have to check whether we adjusted the dates too
				// well so that nothing is printed. That happens if you print just
				// one week which has the last day of a month in it.
				if (date.compare(end) >= 0) {
					date.day = 1;
					try {
						date.normalize();
					} catch(ex) {}
				}
			} else {
				date.day = 1;
				try {
					date.normalize();
				} catch(ex) {}
			}
		} else {
			// If start date is after end date, just print empty month.
			date = realEnd.clone();
		}
	
		while (date.compare(end) < 0) {
			let monthName = cal.l10n.getAnyString("calendar", "dateFormat", "month." + (date.month +1)+ ".name", []);
			monthName += " " + date.year;
	
			let monthsTable = html.createElementNS("http://www.w3.org/1999/xhtml","table");
			monthsTable.setAttribute('border', '0');
			monthsTable.setAttribute('width', '100%');
			monthsTable.className = 'main-table';
			body.appendChild(monthsTable);
				
			let tmpTr = html.createElementNS("http://www.w3.org/1999/xhtml","tr");
			monthsTable.appendChild(tmpTr);
			let printOM = Services.prefs.getBoolPref("fgprinters.monthly.printothermonths", false);
			if (printOM) {
				let prevMonth = date.clone();
				prevMonth.month--;
				prevMonth.day = 15;
				try {
					prevMonth.normalize();
				} catch(ex) {}
				let nextMonth = date.clone();
				nextMonth.month++;
				nextMonth.day = 15;
				try {
					nextMonth.normalize();
				} catch(ex) {}
				let prevMonthTable = this.createMonthTable(html, prevMonth);
				let nextMonthTable = this.createMonthTable(html, nextMonth);
				//			<td width='1%' align='left' valign='center'>{prevMonthTable}</td>
				//			<td width='98%' align='center' valign='center'>{monthName}</td>
				//			<td width='1%' align='right' valign='center'>{nextMonthTable}</td>
				let prevMonthTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				prevMonthTd.setAttribute('width', '1%');
				prevMonthTd.setAttribute('align', 'left');
				prevMonthTd.setAttribute('valign', 'center');
				prevMonthTd.appendChild(prevMonthTable);
				tmpTr.appendChild(prevMonthTd);
				
				let monthNameTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				monthNameTd.setAttribute('width', '98%');
				monthNameTd.setAttribute('align', 'center');
				monthNameTd.setAttribute('valign', 'center');
				monthNameTd.textContent = monthName;
				tmpTr.appendChild(monthNameTd);
				
				let nextMonthTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				nextMonthTd.setAttribute('width', '1%');
				nextMonthTd.setAttribute('align', 'right');
				nextMonthTd.setAttribute('valign', 'center');
				nextMonthTd.appendChild(nextMonthTable);
				tmpTr.appendChild(nextMonthTd);
				
			} else {
				//		<td align='center' valign='center'>{monthName}</td>
				let monthNameTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				monthNameTd.setAttribute('align', 'center');
				monthNameTd.setAttribute('valign', 'center');
				monthNameTd.textContent = monthName;
				tmpTr.appendChild(monthNameTd);
			}
			body.appendChild(this.getStringForMonth(html, date, aItems));
			// Make sure each month gets put on its own page
			//<br style="page-break-after:always;"/>
			let tmpBr = html.createElementNS("http://www.w3.org/1999/xhtml","br");
			tmpBr.setAttribute('style', 'page-break-after:always;');
			body.appendChild(tmpBr);
			date.month++;
			try {
				date.normalize();
			} catch(ex) {}
		}

		// Stream out the resulting HTML
		let htmlString = cal.xml.serializeDOM(html);		
		//Components.utils.reportError(htmlString);
		let convStream = Components.classes["@mozilla.org/intl/converter-output-stream;1"]
                                   .createInstance(Components.interfaces.nsIConverterOutputStream);
		convStream.init(aStream, 'UTF-8', 0, 0x0000);
		convStream.writeString(htmlString);
	},
	
	getStringForMonth : function FGmonthPrint_getHTML(document, aStart, aItems) {
		let weekStart = Services.prefs.getIntPref("calendar.week.start", 0);
		let smallSatSun = Services.prefs.getBoolPref("fgprinters.monthly.smallsatsun", false);
		let printWeekNo = Services.prefs.getBoolPref("fgprinters.monthly.printweekno", false);
	
		//<table style='border:1px solid black;' width='100%' rules='all'/>
		let monthTable = document.createElementNS("http://www.w3.org/1999/xhtml","table");
		monthTable.setAttribute('style', 'border:1px solid black;');
		monthTable.setAttribute('width', '100%');
		monthTable.setAttribute('rules', 'all');
		let dayNameRow = document.createElementNS("http://www.w3.org/1999/xhtml","tr");// <tr/>
		if (printWeekNo) {
			let sbs = Components.classes["@mozilla.org/intl/stringbundle;1"]
								.getService(Components.interfaces.nsIStringBundleService);
			let props = sbs.createBundle("chrome://calendar/locale/calendar.properties");
			let weekName = " " + props.GetStringFromName('shortcalendarweek') + " "; 
			//<td class='day-name' align='center' width='0%'>{weekName}</td>
			let tmpTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			tmpTd.className = 'day-name';
			tmpTd.setAttribute('align', 'center');
			tmpTd.setAttribute('width', '0%');
			tmpTd.textContent = weekName;
			dayNameRow.appendChild(tmpTd);
		}
		if (!smallSatSun) {
			for (let i = 0; i < 7; i++) {
				let dayName = cal.l10n.getAnyString("calendar", "dateFormat", "day."+ (((weekStart+i)%7)+1) + ".Mmm");
				// <td class='day-name' align='center' width='14%'>{dayName}</td>
				let tmpTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
				tmpTd.className = 'day-name';
				tmpTd.setAttribute('align', 'center');
				tmpTd.setAttribute('width', '14%');
				tmpTd.textContent = dayName;
				dayNameRow.appendChild(tmpTd);
			}
		} else {
			for (let i = 0; i < 5; i++) {
				let dayName = ccal.l10n.getCalString("calendar", "dateFormat", "day."+ (((weekStart+i)%7)+1) + ".Mmm");
				//<td class='day-name' align='center' width='16%'>{dayName}</td>
				let tmpTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
				tmpTd.className = 'day-name';
				tmpTd.setAttribute('align', 'center');
				tmpTd.setAttribute('width', '16%');
				tmpTd.textContent = dayName;
				dayNameRow.appendChild(tmpTd);
			}
			let dayName = cal.l10n.getAnyString("calendar", "dateFormat", "day."+ (((weekStart+5)%7)+1) + ".Mmm") + '/' + cal.l10n.getAnyString("calendar", "dateFormat", "day."+ (((weekStart+6)%7)+1) + ".Mmm");
			// <td class='day-name' align='center' width='16%'>{dayName}</td>
			let tmpTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			tmpTd.className = 'day-name';
			tmpTd.setAttribute('align', 'center');
			tmpTd.setAttribute('width', '16%');
			tmpTd.textContent = dayName;
			dayNameRow.appendChild(tmpTd);
		}
		monthTable.appendChild(dayNameRow);
	
		// Set up the item-list so it's easy to work with.
		function hasUsableDate(item) {
			return item.startDate || item.entryDate || item.dueDate;
		}
		let filteredItems = aItems.filter(hasUsableDate);
	
		let calIEvent = Components.interfaces.calIEvent;
		let calITodo = Components.interfaces.calITodo
	
		let defaultTimezone = null;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {}
	
		function compareItems(a, b) {
			// Sort tasks before events
			if (a instanceof calIEvent && b instanceof calITodo) {
				return 1;
			}
			if (a instanceof calITodo && b instanceof calIEvent) {
				return -1;
			}
	
			if (a instanceof calIEvent) {
				if (a.startDate.isDate && b.startDate.isDate) {
					return a.startDate.compare(b.startDate);
				}
				if (a.startDate.isDate) {
					let tmpd = b.startDate.clone();
					tmpd.isDate = true;
					try {
						tmpd.normalize();
					} catch(ex) {}
					if (defaultTimezone) {
						try {
							tmpd = tmpd.getInTimezone(defaultTimezone);
						} catch(ex) {}
					}
					if (a.startDate.compare(tmpd) == 0) return -1;
				}
				if (b.startDate.isDate) {
					let tmpd = a.startDate.clone();
					tmpd.isDate = true;
					try {
						tmpd.normalize();
					} catch(ex) {}
					if (defaultTimezone) {
						try {
							tmpd = tmpd.getInTimezone(defaultTimezone);
						} catch(ex) {}
					}
					if (b.startDate.compare(tmpd) == 0) return 1;
				}
			}
			
			if (a instanceof calIEvent) {
				let startCompare = a.startDate.compare(b.startDate);
				if (startCompare != 0) {
					return startCompare;
				}
				return a.endDate.compare(b.endDate);
			}
			let aDate = a.entryDate || a.dueDate;
			let bDate = b.entryDate || b.dueDate;
			return aDate.compare(bDate);
		}
		let sortedList = filteredItems.sort(compareItems);
		let firstDate = aStart.startOfMonth.startOfWeek.clone();
		firstDate.day += weekStart;
		try {
			firstDate.normalize();
		} catch(ex) {}
		if (aStart.startOfMonth.weekday < weekStart) {
			// Go back one week to make sure we display this day
			firstDate.day -= 7;
			try {
				firstDate.normalize();
			} catch(ex) {}
		}
	
		let lastDate = aStart.endOfMonth.endOfWeek.clone();
		if (aStart.endOfMonth.weekday < weekStart) {
			// Go back one week so we don't display any extra days
			lastDate.day -= 7;
			try {
				lastDate.normalize();
			} catch(ex) {}
		}
		firstDate.isDate = true;
		lastDate.isDate = true;
	
		let date = firstDate.clone();
		//let itemListIndex = 0;
		while (date.compare(lastDate) != 1) {
			monthTable.appendChild(this.makeHTMLWeek(document, date, sortedList, aStart.month));
		}
		return monthTable;
	}, 
	
	makeHTMLDay : function FGmakeHTMLDay(document, date, sortedList, targetMonth, smallDay) {
		let obeyPrivacy = Services.prefs.getBoolPref("fgprinters.obeyprivacy", false);
		const weekPrefix = "calendar.week.";
		let prefNames = ["d0sundaysoff", "d1mondaysoff", "d2tuesdaysoff",
						"d3wednesdaysoff", "d4thursdaysoff", "d5fridaysoff", "d6saturdaysoff"];
		let defaults = [true, false, false, false, false, false, true];
		let daysOff = new Array();
		for (let i in prefNames) {
			if (Services.prefs.getCharPref(weekPrefix+prefNames[i], defaults[i])) {
				daysOff.push(Number(i));
			}
		}
	
		let printAllDay = Services.prefs.getBoolPref("fgprinters.monthly.printallday", false);
		let printPriority = Services.prefs.getBoolPref("fgprinters.monthly.printpriority", true);
		let printStatus = Services.prefs.getBoolPref("fgprinters.monthly.printstatus", true);
		let fsize = Services.prefs.getIntPref("fgprinters.monthly.fontsize", 10);
		let fsizet = Services.prefs.getIntPref("fgprinters.monthly.fontsizetime", 10);
		let oldcolour = Services.prefs.getBoolPref("fgprinters.monthly.oldcolour", false);
		let printLocation = Services.prefs.getBoolPref("fgprinters.monthly.printlocation", false);
		let printDescription = Services.prefs.getBoolPref("fgprinters.monthly.printdescription", true);
		let smallSatSun = Services.prefs.getBoolPref("fgprinters.monthly.smallsatsun", false);
		let printDayLeft = Services.prefs.getBoolPref("fgprinters.monthly.aligndayleft", false);
		let noOtherMonth = Services.prefs.getBoolPref("fgprinters.monthly.noothermonth", false);
		let printEndTime = Services.prefs.getBoolPref("fgprinters.monthly.printendtime", true);
		let noPrintTimesInMultiDay = Services.prefs.getBoolPref("fgprinters.monthly.noprinttimesinmultidayevents", false);
		
		let noEventsToday = noOtherMonth && (date.month != targetMonth);
		let defaultTimezone = null;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {}
	
		let myClass = 'day-box-sun';
		if (smallDay < 2) {
			myClass = 'day-box';
		} 
		if (date.month != targetMonth) {
			myClass += ' out-of-month';
		} else if (daysOff.some(function(a) { return a == date.weekday; })) {
			myClass += ' day-off';
		}
	
		let height = '100%';
	
		if (smallDay) {
			height = '50%';
		}
	
		// <td align='left' valign='top' class={myClass} height={height} width='100'/>
		let dayTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
		dayTd.className = myClass;
		dayTd.setAttribute('align', 'left');
		dayTd.setAttribute('valign', 'top');
		dayTd.setAttribute('height', height);
		dayTd.setAttribute('width', '100');
	
		let tStyle = 'font-size: ' + fsize + 'px;';
	
		//<table width='100%' valign='top' border='0' height={height}/>
		let innerTable =  document.createElementNS("http://www.w3.org/1999/xhtml","table");
		innerTable.setAttribute('width', '100%');
		innerTable.setAttribute('valign', 'top');
		innerTable.setAttribute('border', '0');
		innerTable.setAttribute('height', height);
		if (smallDay >= 2) {
			innerTable.className = myClass;
		}
		let dayNum = date.day;
		if (noEventsToday) {
			dayNum = " ";
		}


		//<tr valign='top' height='1px'>
		//	<td valign='top' align='left' style={tStyle}>{dayNum}</td>
		//</tr>		
		let align = 'right';
		if (printDayLeft) {
			align = 'left';
		}
		let dateLabelTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr");
		dateLabelTr.setAttribute('valign', 'top');
		dateLabelTr.setAttribute('height', '1px');
		let dateLabelTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
		dateLabelTd.setAttribute('align', align);
		dateLabelTd.setAttribute('style', tStyle);
		dateLabelTd.textContent = dayNum;
		dateLabelTr.appendChild(dateLabelTd);
		innerTable.appendChild(dateLabelTr);
		if (!noEventsToday) {
			for (let item of sortedList) {
			let sDate = item.startDate || item.entryDate || item.dueDate;
			let eDate = item.endDate || item.dueDate || item.entryDate;
			if (defaultTimezone && sDate) {
				try {
					sDate = sDate.getInTimezone(defaultTimezone);
				} catch(ex) {}
			}
			if (defaultTimezone && eDate) {
				try {
					eDate = eDate.getInTimezone(defaultTimezone);
				} catch(ex) {}
			}
	
				// end dates are exclusive
			if (sDate.isDate) {
				eDate = eDate.clone();
				eDate.day -= 1;
				try {
					eDate.normalize();
				} catch(ex) {}
			}
			if (!eDate || eDate.compare(date) == -1) {
				continue;
			}
			//itemListIndex = i;
			if (!sDate || sDate.compare(date) == 1) {
				break;
			}
			let dateFormatter = 
				Components.classes["@mozilla.org/calendar/datetime-formatter;1"]
						.getService(Components.interfaces.calIDateTimeFormatter);
	
	
			function getStringForDate(date) {
				let dstring;
				if (!date.isDate) {
					return dateFormatter.formatTime(date);
				}
				if (printAllDay) { return cal.l10n.getCalString("calendar", "dateFormat", "AllDay"); } else { return ""; }
			}
	
			let time;
			if (sDate && eDate) {
				if (!sDate.isDate) {
					let startTime = "";
					let endTime = "";
					if (sDate.compare(date) < 0) {
						let dummyDate = sDate.clone();
						dummyDate.hour = 0;
						dummyDate.minute = 0;
						try {
							dummyDate.normalize();
						} catch(ex) {
						}
						startTime = getStringForDate(dummyDate);
						if (noPrintTimesInMultiDay) {
							startTime = "";
						}
					} else {
						startTime = getStringForDate(sDate);
					}
					if (eDate.compare(date) > 0) {
						let dummyDate = eDate.clone();
						dummyDate.hour = 23;
						dummyDate.minute = 59;
						try {
							dummyDate.normalize();
						} catch(ex) {
						}
						endTime = getStringForDate(dummyDate);
						if (noPrintTimesInMultiDay) {
							endTime = "";
						}
					} else {
						endTime = getStringForDate(eDate);
					}
					if (printEndTime) {
						time = startTime + '-' + endTime;
						if (time == "-") {
							time = "";
						}
					} else {
						time = startTime;
					}
				} else {
					time = getStringForDate(sDate);
				}
			} else if (sDate) {
				time = getStringForDate(sDate);
			} else if (eDate) {
				time = getStringForDate(eDate);
			}
			let calColor = null;
			try {
				calColor = item.calendar.getProperty('color');
			} catch(ex) {
				let calMgr = Components.classes["@mozilla.org/calendar/manager;1"]
									.getService(Components.interfaces.calICalendarManager);
				calColor = calMgr.getCalendarPref(item.calendar, 'color'); 
			}
			if (!calColor) {
				calColor = "#A8C2E1";
			}
			//let pb2 = Components.classes["@mozilla.org/Services.prefs-service;1"]
			//					.getService(Components.interfaces.nsIPrefBranch2);
			let catColor;
			try {
				catColor = Services.prefs.getCharPref("calendar.category.color."+item.getProperty("CATEGORIES").toLowerCase());
			} catch(ex) {}
			if (!catColor) {
				for (let cat of item.getCategories({})) {
					try {
						catColor = Services.prefs.getCharPref("calendar.category.color." + cat.toLowerCase());
						break; // take first matching
					} catch(ex) {}
				}
			}
			let borderType = 'solid';
			if (item.status && printStatus) {
				if (item.status == "TENTATIVE") {
					borderType = 'dotted';
				}
				if (item.status == "CANCELLED") {
					borderType = 'dashed';
				}
			}
			let style = null;
			let tStyle = null;
			if (!oldcolour) {
				style = 'font-size: '+fsize+'px; text-align: left;';
				tStyle = 'font-size: '+fsizet+'px; text-align: left; color: black;';
				style += ' color: black; border: ' + borderType + ' ' + calColor + ' 2px; width: 100%; ';
				if (catColor) {
					style += 'border-right-width: 4px; border-right-color: ' + catColor + ';';
					style += 'border-top-color: ' + catColor + ';';
					catColor = 0;
				}
			} else {
				style = 'font-size: '+fsize+'px; text-align: left; background-color: '+calColor + ';';
				style += ' color: ' + cal.getContrastingTextColor(calColor) + ';';
				tStyle = 'font-size: '+fsizet+'px; text-align: left; background-color: '+calColor + '; color: ' + cal.getContrastingTextColor(calColor) + ';';
				if (catColor) {
					style += ' border: solid ' + catColor + ' 2px;';
					catColor = 0;
				}
			}
			if (printPriority && item.priority && !oldcolour) {
				let prioCol = '';
				if (item.priority > 5) {
					prioCol = "#CFC";
				}
				if (item.priority == 5) {
					prioCol = "#FFC";
				}
				if (item.priority < 5) {
					prioCol = "#FCC";
				}
				style += 'background-color: ' + prioCol + ';';
			}
			let title = item.title;
			let location = " ";
			let descr = " ";
			if (printLocation) {
				if (item.hasProperty('LOCATION')) {
					location = item.getProperty('LOCATION');
				}
			}
			if (printDescription) {
				if (item.hasProperty('DESCRIPTION')) {
					descr = item.getProperty('DESCRIPTION');
				}
			}

			if (obeyPrivacy && (item.privacy == "PRIVATE" || item.privacy == "CONFIDENTIAL")) {
				title = " ";
			}
			//<tr width='100%'>
			//	<td valign='top' align='left' width='100%' style={style}><span style={tStyle}>{time}</span> {title}<div class='desc'>{descr}</div>
			//  <div class='loc'>{location}</div></td>
			//</tr>;
			let itemTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr");
			itemTr.setAttribute('width', '100%');
			let itemTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			itemTd.setAttribute('valign', 'top');
			itemTd.setAttribute('align', 'left');
			itemTd.setAttribute('width', '100%');
			itemTd.setAttribute('style', style);
			itemTr.appendChild(itemTd);
			
			let timeSpan = document.createElementNS("http://www.w3.org/1999/xhtml","span");
			timeSpan.setAttribute('style', tStyle);
			timeSpan.textContent = time;
			itemTd.appendChild(timeSpan);
			
			itemTd.textContent += title;
			
			let descDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div");
			descDiv.className = 'desc';
			descDiv.textContent = descr;
			itemTd.appendChild(descDiv);
			let locDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div");
			locDiv.className = 'loc';
			locDiv.textContent = location;
			itemTd.appendChild(locDiv);
			innerTable.appendChild(itemTr);
		}
	}
		dayTd.appendChild(innerTable);
		if (smallDay > 1) {
			return innerTable;
		} else {
			return dayTd;
		}
	},
	
	makeHTMLWeek : function FGmakeHTMLWeek(document, date, sortedList, targetMonth) {
		let weekRow = document.createElementNS("http://www.w3.org/1999/xhtml","tr");//<tr height='85'/>;
		weekRow.setAttribute('height', '85');
		let obeyPrivacy = Services.prefs.getBoolPref("fgprinters.obeyprivacy", false);
		let printWeekNo = Services.prefs.getBoolPref("fgprinters.monthly.printweekno", false);
		const weekPrefix = "calendar.week.";
		let prefNames = ["d0sundaysoff", "d1mondaysoff", "d2tuesdaysoff",
						"d3wednesdaysoff", "d4thursdaysoff", "d5fridaysoff", "d6saturdaysoff"];
		let defaults = [true, false, false, false, false, false, true];
		let daysOff = new Array();
		for (let i in prefNames) {
			if (Services.prefs.getCharPref(weekPrefix+prefNames[i], defaults[i])) {
				daysOff.push(Number(i));
			}
		}
	
		let printAllDay = Services.prefs.getBoolPref("fgprinters.monthly.printallday", false);
		let printPriority = Services.prefs.getBoolPref("fgprinters.monthly.printpriority", true);
		let printStatus = Services.prefs.getBoolPref("fgprinters.monthly.printstatus", true);
		let fsize = Services.prefs.getIntPref("fgprinters.monthly.fontsize", 10);
		let oldcolour = Services.prefs.getBoolPref("fgprinters.monthly.oldcolour", false);
		let printLocation = Services.prefs.getBoolPref("fgprinters.monthly.printlocation", false);
		let smallSatSun = Services.prefs.getBoolPref("fgprinters.monthly.smallsatsun", false);
	
		if (printWeekNo) {
			let weekFormatter = null;
			try {
				weekFormatter = Components.classes["@mozilla.org/calendar/weektitle-service;1"]
											.getService(Components.interfaces.calIWeekTitleService);
			} catch(ex) {
				weekFormatter = Components.classes["@mozilla.org/calendar/weekinfo-service;1"]
											.getService(Components.interfaces.calIWeekInfoService);
			}
			let weekno = weekFormatter.getWeekTitle(date);
			//<td class='weekno' align='center'> {weekno} </td>;
			let weekNoTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			weekNoTd.className = 'weekno';
			weekNoTd.setAttribute('align', 'center');
			weekNoTd.textContent = weekno;
			weekRow.appendChild(weekNoTd);
		}
	
		let defaultTimezone = null;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {}
		if (!smallSatSun) {
			for (let i = 0; i < 7; i++) {
				weekRow.appendChild(this.makeHTMLDay(document, date, sortedList, targetMonth, 0));
				date.day++;
				try {
					date.normalize();
				} catch(ex) {}
			}
		} else {
			for (let i = 0; i < 5; i++) {
				weekRow.appendChild(this.makeHTMLDay(document, date, sortedList, targetMonth, 0));
				date.day++;
				try {
					date.normalize();
				} catch(ex) {}
			}
			let sat = this.makeHTMLDay(document, date, sortedList, targetMonth, 1);
			date.day++;
			try {
				date.normalize();
			} catch(ex) {}
			let sun = this.makeHTMLDay(document, date, sortedList, targetMonth, 2);
			date.day++;
			try {
				date.normalize();
			} catch(ex) {}
			sat.appendChild(sun);
			weekRow.appendChild(sat);
		}
		return weekRow;
	},
	
	createMonthTable : function FGmonthPrint_doOtherMonth(document, aMonth) {
		let weekStart = Services.prefs.getIntPref("calendar.week.start", 0);	
		let monthName = cal.l10n.getCalString("calendar", "dateFormat", "month." + (aMonth.month +1)+ ".name") + " " + aMonth.year;
		//let monthName = cal.l10n.formatMonth(startOfMonth.month + 1, "calendar", "monthInYear") + " " + aMonth.year;
	
		let fsize = Services.prefs.getIntPref("fgprinters.monthly.fontsizeom", 10);
		let tStyle = 'font-size: ' + fsize + 'px;';
		//<table width='100%' rules='both' style={tStyle}><tr><td colspan='7' align='center'>{monthName}</td></tr></table>;
		let monthTable = document.createElementNS("http://www.w3.org/1999/xhtml","table");
		monthTable.setAttribute('width', '100%');
		monthTable.setAttribute('rules', 'both');
		monthTable.setAttribute('style', tStyle);
		let tmpTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr");
		monthTable.appendChild(tmpTr);
		let tmpTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
		tmpTd.setAttribute('colspan', '7');
		tmpTd.setAttribute('align', 'center');
		tmpTd.textContent = monthName;
		tmpTr.appendChild(tmpTd);
		
		let firstDate = aMonth.startOfMonth.startOfWeek.clone();
		firstDate.day += weekStart;
		try {
			firstDate.normalize();
		} catch(ex) {}
		if (aMonth.startOfMonth.weekday < weekStart) {
			// Go back one week to make sure we display this day
			firstDate.day -= 7;
			try {
				firstDate.normalize();
			} catch(ex) {}
		}

		let lastDate = aMonth.endOfMonth.endOfWeek.clone();
		if (aMonth.endOfMonth.weekday < weekStart) {
			// Go back one week so we don't display any extra days
			lastDate.day -= 7;
			try {
				lastDate.normalize();
			} catch(ex) {}
		}
		firstDate.isDate = true;
		lastDate.isDate = true;
	
		let printWeekDays = Services.prefs.getBoolPref("fgprinters.printweekdaysmm", false);
		if (printWeekDays) {
			const weekPrefix = "calendar.week.";
			let prefNames = ["d0sundaysoff", "d1mondaysoff", "d2tuesdaysoff",
							"d3wednesdaysoff", "d4thursdaysoff", "d5fridaysoff", "d6saturdaysoff"];
			let defaults = [true, false, false, false, false, false, true];
			let daysOff = new Array();
			for (let i in prefNames) {
				if (Services.prefs.getCharPref(weekPrefix+prefNames[i], defaults[i])) {
					daysOff.push(Number(i));
				}
			}
			let weekRow = document.createElementNS("http://www.w3.org/1999/xhtml","tr");//<tr/>;
			for (let i = 0; i < 7; i++) {
				let myClass = 'day-box';
				let style = 'padding-left: 0.2em; padding-right: 0.2em; ';
				let dayName = cal.l10n.getAnyString("calendar", "dateFormat", "day."+ (((i + weekStart) % 7)+1) + ".Mmm")[0];
				//<td align='center' valign='center' class={myClass}><div style={style}>{dayName}</div></td>
				let day = document.createElementNS("http://www.w3.org/1999/xhtml","td");
				day.setAttribute('align', 'center');
				day.setAttribute('valign', 'center');
				day.className = myClass;
				let tmpDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div");
				tmpDiv.setAttribute('style', style);
				tmpdiv.textContent = dayName;
				day.appendChild(tmpDiv);
				weekRow.appendChild(day);
				try {
					date.normalize();
				} catch(ex) {}
			}
			monthTable.appendChild(weekRow);
		}

		let date = firstDate.clone();
		//let itemListIndex = 0;
		while (date.compare(lastDate) != 1) {
			monthTable.appendChild(this.makeOtherMonthWeek(document, date, aMonth.month));
		}
		return monthTable;
	},

	makeOtherMonthWeek : function FGmakeOtherMonthWeek(document, date, targetMonth) {
		let weekRow = document.createElementNS("http://www.w3.org/1999/xhtml","tr");//<tr/>;
		let printOMDays = Services.prefs.getBoolPref("fgprinters.printomdaysmm", true);
		const weekPrefix = "calendar.week.";
		let prefNames = ["d0sundaysoff", "d1mondaysoff", "d2tuesdaysoff",
						"d3wednesdaysoff", "d4thursdaysoff", "d5fridaysoff", "d6saturdaysoff"];
		let defaults = [true, false, false, false, false, false, true];
		let daysOff = new Array();
		for (let i in prefNames) {
			if (Services.prefs.getCharPref(weekPrefix+prefNames[i], defaults[i])) {
				daysOff.push(Number(i));
			}
		}

		for (let i = 0; i < 7; i++) {
			let style = 'padding-left: 0.2em; padding-right: 0.2em;';
			let myClass = 'day-box';
			if (date.month != targetMonth) {
				myClass += ' out-of-month';
				if (!printOMDays) {
					style += 'color: transparent; ';
				}
			} else if (daysOff.some(function(a) { return a == date.weekday; })) {
				myClass += ' day-off';
			}
			// <td align='right' valign='center' class={myClass}><div style={style}>{date.day}</div></td>
			let day = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			day.setAttribute('align', 'right');
			day.setAttribute('valign', 'center');
			day.className = myClass;
			let tmpDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div");
			tmpDiv.setAttribute('style', style);
			tmpdiv.textContent = date.day;
			day.appendChild(tmpDiv);
			weekRow.appendChild(day);
			date.day++;
			try {
				date.normalize();
			} catch(ex) {}
		}
		return weekRow;
	},
	
	setupStyleSheet: function FGlist_setupStyleSheet(html) {
		let fontsizetitle = Services.prefs.getIntPref("fgprinters.monthly.fontsizetitle", 26);
		let fontsizeday = Services.prefs.getIntPref("fgprinters.monthly.fontsizeday", 12);
		let dnbk = Services.prefs.getCharPref("fgprinters.headcolour", "#f0f0f0");
		let oomc = Services.prefs.getCharPref("fgprinters.othermonthcolour", "#EEEEFF");
		let wec = Services.prefs.getCharPref("fgprinters.weekendcolour", "#FFFFEE");
		let sansserif = Services.prefs.getBoolPref("fgprinters.sansserif", true);
		let fsizeloc = Services.prefs.getIntPref("fgprinters.monthly.fontsizelocation", 8);
		let fsizedesc = Services.prefs.getIntPref("fgprinters.monthly.fontsizedescription", 8);
		let printLocationRight = Services.prefs.getBoolPref("fgprinters.monthly.printlocationright", true);
		let fsizeglobaltitle = Services.prefs.getIntPref("fgprinters.fontsizetitle", 26);
		let fsizesubtitle = Services.prefs.getIntPref("fgprinters.fontsizesubtitle", 20);
		let fsizeweekno = Services.prefs.getIntPref("fgprinters.monthly.fontsizeweekno", 12);
		
		let style = ".main-table { font-size: "+fontsizetitle+"px; font-weight: bold; }\n";
		style += ".sub-title { font-size: "+fsizesubtitle+"px; font-weight:bold; text-align: center; }\n";
		style += ".main-title { font-size: "+fsizeglobaltitle+"px; font-weight:bold; text-align: center; }\n";
		style += ".day-name { border: 1px solid black; background-color: "+dnbk+"; font-size: "+fontsizeday+"px; font-weight: bold; }\n";
		style += ".day-box { border: 1px solid black; vertical-align: top; }\n";
		style += ".day-box-sun { border-top: 1px solid black; vertical-align: top; }\n";
		style += ".out-of-month { background-color: "+oomc+" !important; }\n";
		style += ".day-off { background-color: "+wec+" !important; }\n";
		style += ".weekno { font-size: "+fsizeweekno+"px; }\n";
	
		if (printLocationRight) {
			style += ".loc { font-size: "+fsizeloc+"px; text-align: right; }\n";
		} else {
			style += ".loc { font-size: "+fsizeloc+"px; text-align: left; }\n";
		}
		style += ".desc { font-size: "+fsizedesc+"px; text-align: left; }\n";
	
		if (sansserif) { style += "body, table, tr, td, p { font-family: Arial, sans-serif; }\n"; }
		html.getElementsByTagName('style')[0].textContent += "\n" + style;
	}
}