/* -*- Mode: javascript; tab-width: 20; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Calendar code.
 *
 * The Initial Developer of the Original Code is
 *   Joey Minta <jminta@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * The output code for the time sheet was written by
 *   Ferdinand Grassmann <ferdinand@grassmann.info>
 *
 * Contributor(s):
 *   Matthew Willis <lilmatt@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

var { Services } = ChromeUtils.import("resource://gre/modules/Services.jsm");
var { cal } = ChromeUtils.import("resource://calendar/modules/calUtils.jsm");
 
Components.utils.import("resource://calendar/modules/calUtils.jsm");
//Components.utils.import("resource://fgprinters/modules/calXMLUtils.jsm");
//Components.utils.import("resource://fgprinters/modules/calPrintUtils.jsm");
Components.utils.import("resource://fgprinters/modules/calHtmlUtils.jsm");
  
/**
 * Prints a time sheet view of a week of events
 */
function FGcalTimeSheetPrinter() {
}

var FGcalTimeSheetPrinterID = Components.ID('{9083C5EC-5213-11DC-BF8C-418855D89593}');
var FGcalTimeSheetPrinterInterfaces = [Components.interfaces.calIPrintFormatter];
var FGcalTimeSheetPrinterContractID = "@mozilla.org/calendar/printformatter;1?type=fgtimesheet";
var FGcalTimeSheetPrinterDescription = "Calendar Week TimeSheet Print Formatter";

FGcalTimeSheetPrinter.prototype = {
	classID: FGcalTimeSheetPrinterID,
	QueryInterface: ChromeUtils.generateQI([Ci.calIPrintFormatter]),
	
		
		get name() {
        return "FG " + cal.l10n.getAnyString("fgprinters", "fgprinters", "timesheet", []); 
    },

	formatToHtml : function FGTimeSheetPrint_format(aStream, aStart, aEnd, aCount, aItems, aTitle) {
		
		// Parse the template file
		let html = cal.xml.parseFile("chrome://fgprinters/skin/printing/empty.html");
	
		// Set page title
		html.getElementsByTagName("title")[0].textContent = aTitle;
	
		// Set the style sheet
		this.setupStyleSheet(html);

		let seperateDays = Services.prefs.getBoolPref("fgprinters.ts.seperatedays", false);
		let printHourLines = Services.prefs.getBoolPref("fgprinters.ts.printhourlines", true);
		let rowsSameHeight = Services.prefs.getBoolPref("fgprinters.ts.rowssameheight", false);
		
		let body = html.getElementsByTagName('body')[0];

		// Title and Subtitle
		let printtitle = Services.prefs.getBoolPref("fgprinters.printtitle", false);
		let subTitle = Services.prefs.getCharPref("fgprinters.subtitle", "");
		
		//Components.utils.reportError('PT: '+printtitle+'  , '+subTitle);
		
		if (printtitle && aTitle) {
			let p_aTitle = html.createElementNS("http://www.w3.org/1999/xhtml","p");
			p_aTitle.className = 'main-title';
			p_aTitle.textContent = aTitle;
			body.appendChild(p_aTitle);
		}
		if (subTitle) {
			let p_subTitle = html.createElementNS("http://www.w3.org/1999/xhtml","p");
			p_subTitle.className = 'sub-title';
			p_subTitle.textContent = subTitle;
			body.appendChild(p_subTitle);
		}
				
		// helper: returns the passed item's startDate, entryDate or dueDate, in
		//         that order. If the item doesn't have one of those dates, this
		//         doesn't return.
		function hasUsableDate(item) {
				return item.startDate || item.entryDate || item.dueDate;
		}
		
		// Clean out the item list so it only contains items we will want to
		// include in the printout.
		let filteredItems = aItems.filter(hasUsableDate);

		let calIEvent = Components.interfaces.calIEvent;
		let calITodo = Components.interfaces.calITodo;

		let defaultTimezone = null;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {}
		
		function compareItems(a, b) {
			// Sort tasks before events
			if (a instanceof calIEvent && b instanceof calITodo) {
					return 1;
			}
			if (a instanceof calITodo && b instanceof calIEvent) {
				return -1;
			}

			if (a instanceof calIEvent) {
				if (a.startDate.isDate && b.startDate.isDate) {
					return a.startDate.compare(b.startDate);
				}
				if (a.startDate.isDate) {
					let tmpd = b.startDate.clone();
					tmpd.isDate = true;
					try {
						tmpd.normalize();
					} catch(ex) {}
					if (defaultTimezone) {
						try {
							tmpd = tmpd.getInTimezone(defaultTimezone);
						} catch(ex) {}
					}
					if (a.startDate.compare(tmpd) == 0) return -1;
				}
				if (b.startDate.isDate) {
					let tmpd = a.startDate.clone();
					tmpd.isDate = true;
					try {
						tmpd.normalize();
					} catch(ex) {
					}
					if (defaultTimezone) {
						try {
							tmpd = tmpd.getInTimezone(defaultTimezone);
						} catch(ex) {}
					}
					if (b.startDate.compare(tmpd) == 0) return 1;
				}
			}
			
			if (a instanceof calIEvent) {
				let startCompare = a.startDate.compare(b.startDate);
				if (startCompare != 0) {
					return startCompare;
				}
				return a.endDate.compare(b.endDate);
			}
			let aDate = a.entryDate || a.dueDate;
			let bDate = b.entryDate || b.dueDate;
			return aDate.compare(bDate);
		}
		let sortedList = filteredItems.sort(compareItems);
	
		let sbs = Components.classes["@mozilla.org/intl/stringbundle;1"]
							.getService(Components.interfaces.nsIStringBundleService);
		let props = sbs.createBundle("chrome://calendar/locale/calendar.properties");
		let weekFormatter = null;
		try {
			weekFormatter = Components.classes["@mozilla.org/calendar/weektitle-service;1"]
										.getService(Components.interfaces.calIWeekTitleService);
		} catch(ex) {
			weekFormatter = Components.classes["@mozilla.org/calendar/weekinfo-service;1"]
										.getService(Components.interfaces.calIWeekInfoService);
		}
	
		// A place to cache the string for "All Day" events later if we need it.
		let mAllDayString = null;
		
		// Start at the beginning of the week that aStart is in, and loop until
		// we're at aEnd. In the loop we build the HTML table for each day, and
		// get the day's items using getDayTd().
		let start = aStart || sortedList[0].startDate || sortedList[0].entryDate ||
					sortList[0].dueDate;
		cal.ASSERT(start, "can't find a good starting date to print");
	
		let lastItem = sortedList[sortedList.length-1];
		let end = aEnd || lastItem.startDate || lastItem.entryDate ||
				lastItem.dueDate;
		cal.ASSERT(end, "can't find a good ending date to print");
	
		let date;
	
		let startOfWeek = Services.prefs.getIntPref("calendar.week.start", 0);
		if (!seperateDays) {
			date = start.startOfWeek;
			date.day += startOfWeek;
			try {
				date.normalize();
			} catch(ex) {}
			// Make sure we didn't go too far ahead
			if (date.compare(start) == 1) {
				date.day -= 7;
				try {
					date.normalize();
				} catch(ex) {}
			}
		} else {
			date = start.clone();
		}
		
		const weekPrefix = "calendar.week.";
		let prefNames = ["d0sundaysoff", "d1mondaysoff", "d2tuesdaysoff",
						"d3wednesdaysoff", "d4thursdaysoff", "d5fridaysoff", "d6saturdaysoff"];
		let defaults = [true, false, false, false, false, false, true];
	
		if (!seperateDays) {
			while(date.compare(end) == -1) {
				let weekno = weekFormatter.getWeekTitle(date);
				let weekTitle = props.formatStringFromName('WeekTitle', [weekno], 1);
				// <table rules='cols' width='50px' class='main-table'>
				//				<tr> 
				//					<td align='center' valign='bottom' colspan='99'>{weekTitle}</td>
				//				</tr>
				//			</table>;
				let mainWeek =  html.createElementNS("http://www.w3.org/1999/xhtml","table");
				mainWeek.className = 'main-table';
				mainWeek.setAttribute('rules', 'cols');
				mainWeek.setAttribute('width', '50px');
				let mainWeekTr = html.createElementNS("http://www.w3.org/1999/xhtml","tr");
				mainWeek.appendChild(mainWeekTr);
				let mainWeekTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				mainWeekTd.setAttribute('align', 'center');
				mainWeekTd.setAttribute('valign', 'bottom');
				mainWeekTd.setAttribute('colspan', '99');
				mainWeekTd.textContent = weekTitle;
				mainWeekTr.appendChild(mainWeekTd);
		
				// Create the <td> for each day, and put it into an array.
				let dayTds = new Array();
				let dayTopTds = new Array();
				for (let i = 0; i < 7 ; i++) {
					let isOff = Services.prefs.getCharPref(weekPrefix+prefNames[date.weekday], defaults[date.weekday]);
					dayTds[date.weekday] = this.getDayTd(html, date, sortedList, isOff);
					dayTopTds[date.weekday] = this.getAllDayTd(html, date, sortedList, isOff);
					date.day += 1;
					try {
						date.normalize();
					} catch(ex) {}
				}
	
				let fsize = Services.prefs.getIntPref("fgprinters.ts.fontsize", 12);
			
			// <td ...?
				let col = html.createElementNS("http://www.w3.org/1999/xhtml","td"); // <td valign="top"/>
				col.setAttribute('valign', 'top');
				let startDay = Services.prefs.getIntPref("fgprinters.ts.starthour", 7);
				let displayHours = Services.prefs.getIntPref("fgprinters.ts.displayhours", 15);
				let endDay = startDay + displayHours;
				if (endDay > 24) {
					endDay = 24;
					displayHours = endDay - startDay;
				}

				let posStyle = 'position: relative; top: 0px; color: #000; font-size: ' + fsize + 'px';
				let posStyleWhite = 'position: relative; top: 0px; color: #fff; font-size: ' + fsize + 'px; width: 3em;';
				//<table valign='top' border='0'><tr height='5px'></tr></table>;
				let myTab = html.createElementNS("http://www.w3.org/1999/xhtml","table"); 
				myTab.setAttribute('valign', 'top');
				myTab.setAttribute('border', '0');
				let myTr = html.createElementNS("http://www.w3.org/1999/xhtml","tr"); 
				myTr.setAttribute('height', '5px');
				myTab.appendChild(myTr);
				let myTd = html.createElementNS("http://www.w3.org/1999/xhtml","td"); //<td valign="top"/>
				myTd.setAttribute('valign', 'top');
				let myDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div"); //<div style={posStyle}> </div>;
				myDiv.setAttribute('style', posStyle);
				let myWhiteDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div"); //<div style={posStyleWhite}> </div>
				myWhiteDiv.setAttribute('style', posStyleWhite);
				myDiv.appendChild(myWhiteDiv); 
	
				myTd.appendChild(myDiv);
				myTr.appendChild(myTd);
				col.appendChild(myTab);
				mainWeek.appendChild(col);
	
				// print allday-events
				for (let i = startOfWeek; i < 7 + startOfWeek; i++) {
					mainWeek.appendChild(dayTopTds[i % 7]);
				}
			
				// print horizontal lines and times
				if (printHourLines && (displayHours > 1)) {
					let myLines = html.createElementNS("http://www.w3.org/1999/xhtml","tr");//<tr height="0px"><td colspan="99"></td></tr>;
					myLines.setAttribute('height', '0px');
					let myLinesTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
					myLinesTd.setAttribute('colspan', '99');
					myLines.appendChild(myLinesTd);
					let style='position: relative; top: 4px; font-size: ' + fsize + 'px; font-weight: normal;';
					let myTimes = html.createElementNS("http://www.w3.org/1999/xhtml","div");//<div style={style}> </div>;
					myTimes.setAttribute('style', style);
					for (let horLines = 0; horLines < displayHours; horLines++) {
						let myPosition = ((horLines * 500) / displayHours + horLines*0) - 2;
						let style= 'position: absolute; top: ' + myPosition + 'px; width: 100%; border-top: solid #aaa 1px; ';
						let oneLine = html.createElementNS("http://www.w3.org/1999/xhtml","div"); //<div style={style}> </div>;
						oneLine.setAttribute('style', style);
						myTimes.appendChild(oneLine);
					}
	
					for (let times = startDay; times < endDay; times++) {
						let timePos = ((times - startDay) * 500) / displayHours;
						let style='position: absolute; top: ' + timePos + 'px; font-size: ' + fsize + 'px; font-weight: normal;';
						let hourDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div"); //<div style={style}>{times}:00</div>
						hourDiv.setAttribute('style', style);
						hourDiv.textContent = times + ':00';
						myTimes.appendChild(hourDiv);
					} 
					myLinesTd.appendChild(myTimes);
					mainWeek.appendChild(myLines); 
				}
			
				// print non-allday-events for the day
				//let bottomTable = <tr><div style="position: relative; top: 0px;"></div></tr>;
				let bottomTable = html.createElementNS("http://www.w3.org/1999/xhtml","tr");
				let bottomTableTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				bottomTable.appendChild(bottomTableTd);
				for (let i = startOfWeek; i < 7 + startOfWeek; i++) {
					bottomTable.appendChild(dayTds[i % 7]);
				}
				mainWeek.appendChild(bottomTable);
	
				body.appendChild(mainWeek);
				// Make sure each month gets put on its own page
				let tmpBr = html.createElementNS("http://www.w3.org/1999/xhtml","br"); //<br style="page-break-after: always;"/>
				tmpBr.setAttribute('style', 'page-break-after: always;');
				body.appendChild(tmpBr);
			}
		} else {
			while(date.compare(end) == -1) {
				let isOff = Services.prefs.getCharPref(weekPrefix+prefNames[date.weekday], defaults[date.weekday]);
				let dayTd = this.getDayTd(html, date, sortedList, isOff);
				let dayTopTd = this.getAllDayTd(html, date, sortedList, isOff);
				date.day += 1;
				try {
					date.normalize();
				} catch(ex) {}
	
				let weekno = weekFormatter.getWeekTitle(date);
				let weekTitle = props.formatStringFromName('WeekTitle', [weekno], 1);
	
				//<table border='1' rules='cols' width='100%' class='main-table'>
				//				<tr> 
				//					<td align='center' valign='bottom' colspan='99'>{weekTitle}</td>
				//				</tr>
				//			</table>;
				let mainWeek =  html.createElementNS("http://www.w3.org/1999/xhtml","table");
				mainWeek.className = 'main-table';
				mainWeek.setAttribute('border', '1');
				mainWeek.setAttribute('rules', 'cols');
				mainWeek.setAttribute('width', '100%');
				let mainWeekTr = html.createElementNS("http://www.w3.org/1999/xhtml","tr");
				mainWeek.appendChild(mainWeekTr);
				let mainWeekTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
				mainWeekTd.setAttribute('align', 'center');
				mainWeekTd.setAttribute('valign', 'bottom');
				mainWeekTd.setAttribute('colspan', '99');
				mainWeekTd.textContent = weekTitle;
				mainWeekTr.appendChild(mainWeekTd);
							
	
				let fsize = Services.prefs.getIntPref("fgprinters.ts.fontsize", 12);
			
			// <td ...?
				let col = html.createElementNS("http://www.w3.org/1999/xhtml","td"); // <td valign="top"/>
				col.setAttribute('valign', 'top');
				let startDay = Services.prefs.getIntPref("fgprinters.ts.starthour", 7);
				let displayHours = Services.prefs.getIntPref("fgprinters.ts.displayhours", 15);
				let endDay = startDay + displayHours;
				if (endDay > 24) {
					endDay = 24;
					displayHours = endDay - startDay;
				}
	
				let posStyle = 'position: relative; top: 0px; color: #000; font-size: ' + fsize + 'px';
				let posStyleWhite = 'position: relative; top: 0px; color: #fff; font-size: ' + fsize + 'px; width: 3em;';
				//<table valign='top' border='0'><tr height='5px'></tr></table>;
				let myTab = html.createElementNS("http://www.w3.org/1999/xhtml","table");
				myTab.setAttribute('valign', 'top');
				myTab.setAttribute('border', '0');
				let myTr = html.createElementNS("http://www.w3.org/1999/xhtml","tr"); 
				myTr.setAttribute('height', '5px');
				myTab.appendChild(myTr);
				let myTd = html.createElementNS("http://www.w3.org/1999/xhtml","td"); //<td valign="top"/>
				myTd.setAttribute('valign', 'top');
				let myDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div"); //<div style={posStyle}> </div>;
				myDiv.setAttribute('style', posStyle);
				let myWhiteDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div"); //<div style={posStyleWhite}> </div>
				myWhiteDiv.setAttribute('style', posStyleWhite);
				myDiv.appendChild(myWhiteDiv); 
				myTd.appendChild(myDiv);
				myTab.appendChild(myTd);
				col.appendChild(myTab);
			
				mainWeek.appendChild(col);
			
				// print allday-events
				mainWeek.appendChild(dayTopTd);
			
				// print horizontal lines and times
	
				if (printHourLines && (displayHours > 1)){
					let myLines = html.createElementNS("http://www.w3.org/1999/xhtml","tr");//<tr height="0px"><td colspan="99"></td></tr>;
					myLines.setAttribute('height', '0px');
					let myLinesTd = html.createElementNS("http://www.w3.org/1999/xhtml","td");
					myLinesTd.setAttribute('colspan', '99');
					myLines.appendChild(myLinesTd);
					let style='position: relative; top: 4px; font-size: ' + fsize + 'px; font-weight: normal;';
					let myTimes = html.createElementNS("http://www.w3.org/1999/xhtml","div");//<div style={style}> </div>;
					myTimes.setAttribute('style', style);
					for (let horLines = 0; horLines < displayHours; horLines++){
						let myPosition = ((horLines * 500) / displayHours + horLines*0) - 2;
						let style= 'position: absolute; top: ' + myPosition + 'px; width: 100%; border-top: solid #aaa 1px; ';
						let oneLine = html.createElementNS("http://www.w3.org/1999/xhtml","div");//<div style={style}> </div>;
						oneLine.setAttribute('style', style);
						myTimes.appendChild(oneLine);
					}
	
					for (let times = startDay; times < endDay; times++){
						let timePos = ((times - startDay) * 500) / displayHours;
						let style='position: absolute; top: ' + timePos + 'px; font-size: ' + fsize + 'px; font-weight: normal;';
						let hourDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div");
						hourDiv.setAttribute('style', style);
						hourDiv.textContent = times + ':00';
						myTimes.appendChild(hourDiv);
					} 
					myLines.appendChild(myTimes);
					mainWeek.appendChild(myLines); 
				}
			
				// print non-allday-events for the day
				//<tr><div style="position: relative; top: 0px;"></div></tr>;
				let bottomTable = html.createElementNS("http://www.w3.org/1999/xhtml","tr");
				let bottomTableDiv = html.createElementNS("http://www.w3.org/1999/xhtml","div");
				bottomTablediv.setAttribute('style', 'position: relative; top: 0px;');
				bottomTable.appendchild(bottomTableDiv);
				
				bottomTable.appendChild(dayTd);
				mainWeek.appendChild(bottomTable);
	
				body.appendChild(mainWeek);
				// Make sure each month gets put on its own page
				let tmpBr = html.createElementNS("http://www.w3.org/1999/xhtml","br"); //<br style="page-break-after: always;"/>
				tmpBr.setAttribute('style', 'page-break-after: always;');
				body.appendChild(tmpBr);
			}
		}
		// Stream out the resulting HTML
		let htmlString = cal.xml.serializeDOM(html);		
		Components.utils.reportError(htmlString);
		let convStream = Components.classes["@mozilla.org/intl/converter-output-stream;1"]
                                   .createInstance(Components.interfaces.nsIConverterOutputStream);
		convStream.init(aStream, 'UTF-8', 0, 0x0000);
		convStream.writeString(htmlString);
	},

	/**
	 * Given a calIDateTime and an array of items, this function creates an HTML
     * table containing the items, using the appropriate formatting and colours.
     */
	getAllDayTd : function FGTimeSheetPrint_getAllDayTable(document, aDate, aItems, isOff) {
		// mainTd is the <td> element from the parent HTML table that will hold
		// the child HTML tables containing the date string and this day's items.
		let mainTd = document.createElementNS("http://www.w3.org/1999/xhtml","td"); //<td border='0' width='100%' valign='top'/>
		mainTd.setAttribute('border', '0');
		mainTd.setAttribute('width', '100%');
		mainTd.setAttribute('valign', 'top');
		let dateFormatter = Components.classes["@mozilla.org/calendar/datetime-formatter;1"]
                                  .getService(Components.interfaces.calIDateTimeFormatter);
		let defaultTimezone = null;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {}
		let dateString = null;
		try {
			dateString = dateFormatter.dayName(aDate.getInTimezone(defaultTimezone).weekday) + " " + dateFormatter.formatDateShort(aDate.getInTimezone(defaultTimezone));
		} catch(ex) {
			dateString = dateFormatter.dayName(aDate.weekday) + " " + dateFormatter.formatDateShort(aDate);
		}
		// Add the formatted date string (in its own child HTML table)
		//<table class='day-name' width='100%'>
        //                 <tr> 
        //                     <td align='center' valign='bottom'>{dateString}<div style='width: 130px;'> </div></td>
        //                 </tr>
        //             </table>
		let daynameTable = document.createElementNS("http://www.w3.org/1999/xhtml","table"); 
		daynameTable.className = 'day-name';
		daynameTable.setAttribute('width', '100%');
		let daynameTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr"); 
		daynameTable.appendChild(daynameTr);
		let daynameTd = document.createElementNS("http://www.w3.org/1999/xhtml","td"); 
		daynameTd.setAttribute('align', 'center');
		daynameTd.setAttribute('valign', 'bottom');
		daynameTr.appendChild(daynameTd);
		daynameTd.textContent = dateString;
		let tmpDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div"); 
		tmpDiv.setAttribute('style', 'width: 130px;');
		daynameTd.appendChild(tmpDiv);
		
		mainTd.appendChild(daynameTable);

		let printAllDay = Services.prefs.getBoolPref("fgprinters.ts.printallday", false);
		let printLocation = Services.prefs.getBoolPref("fgprinters.ts.printlocation", true);
		let printDescription = Services.prefs.getBoolPref("fgprinters.ts.printdescription", true);
		let printPriority = Services.prefs.getBoolPref("fgprinters.ts.printpriority", true);
		let printStatus = Services.prefs.getBoolPref("fgprinters.ts.printstatus", true);
		let printEndTime = Services.prefs.getBoolPref("fgprinters.ts.printendtime", true);
		let obeyPrivacy = Services.prefs.getBoolPref("fgprinters.obeyprivacy", true);
		let startDay = Services.prefs.getIntPref("fgprinters.ts.starthour", 7);
		let displayHours = Services.prefs.getIntPref("fgprinters.ts.displayhours", 15);
		let noPrintTimesInMultiDay = Services.prefs.getBoolPref("fgprinters.ts.noprinttimesinmultidayevents", false);
	
		let endDay = startDay + displayHours;
		if (endDay > 24) {
			Components.utils.reportError("Too many hours to display (" + endDay + "), adjusting...");
			endDay = 24;
			displayHours = endDay - startDay;
		}
	
		let fsize = Services.prefs.getIntPref("fgprinters.ts.fontsize", 12);
		let fsizet = Services.prefs.getIntPref("fgprinters.ts.fontsizetime", 12);
	
		let oldcolour = Services.prefs.getBoolPref("fgprinters.ts.oldcolour", false);
	
		function getStringForDate1(aDate) {
			if (!aDate.isDate) {
				return dateFormatter.formatTime(aDate);
			}
	
			// We cache the string for "All Day" 'cause we're good like that.
            if (this.mAllDayString == null) {
              let sbs = Components.classes["@mozilla.org/intl/stringbundle;1"]
                                    .getService(Components.interfaces.nsIStringBundleService);
                let props = sbs.createBundle("chrome://calendar/locale/dateFormat.properties");
                this.mAllDayString = props.GetStringFromName("AllDay");
            }
            if (printAllDay) { return this.mAllDayString; } else { return ""; }
        }

        function isEventInRange1(item) {
            let sDate = item.startDate || item.entryDate || item.dueDate;
            let eDate = item.endDate || item.dueDate || item.entryDate;
            if ((sDate.compare(aDate) <= 0) && (eDate.compare(aDate) >= 0)) {
                return 1;
            }
        }

		let filteredEvents = aItems.filter(isEventInRange1);
		//<table width='100%' valign='top' class="weekend" />; }
		let topTable = document.createElementNS("http://www.w3.org/1999/xhtml","table"); 
		topTable.setAttribute('width', '100%');
		topTable.setAttribute('valign', 'top');
		if (isOff) { 
			topTable.className = 'weekend';  
		}
    
		for (let item of filteredEvents) {
			let sDate = item.startDate || item.entryDate || item.dueDate;
			let eDate = item.endDate || item.dueDate || item.entryDate;
			if (sDate) {
				try {
					sDate = sDate.getInTimezone(defaultTimezone);
				} catch(ex) {}
			}
			if (eDate) {
				try {
					eDate = eDate.getInTimezone(defaultTimezone);
				} catch(ex) {}
			}
			
			// End dates are exclusive. Adjust the eDate accordingly.
			if (sDate && sDate.isDate && eDate) {
				eDate = eDate.clone();
				eDate.day -= 1;
				try {
					eDate.normalize();
				} catch(ex) {}
			}
	
			// If the item has no end date, or if the item's end date is aDate or
			// is before aDate, skip to the next item.
			if (!eDate || (eDate.compare(aDate) < 0)) {
				continue;
			}
	
			// No start date or a start date that's after the date we want is bad.
			if (!sDate || (sDate.compare(aDate) > 0)) {
				break;
			}
			
			// only all day events!
			
			if ((sDate && !sDate.isDate) || (eDate && !eDate.isDate)) {
				continue;
			}
	
			let time;
			if (sDate && eDate) {
				if (!sDate.isDate) {
					let startTime = "";
					let endTime = "";
					if (sDate.compare(aDate) < 0) {
						let dummyDate = sDate.clone();
						dummyDate.hour = 0;
						dummyDate.minute = 0;
						try {
							dummyDate.normalize();
						} catch(ex) {}
						startTime = getStringForDate1(dummyDate);
						if (noPrintTimesInMultiDay) {
							startTime = "";
						}
					} else {
						startTime = getStringForDate1(sDate);
					}
					if (eDate.compare(aDate) > 0) {
						let dummyDate = eDate.clone();
						dummyDate.hour = 23;
						dummyDate.minute = 59;
						try {
							dummyDate.normalize();
						} catch(ex) {
						}
						endTime = getStringForDate1(dummyDate);
						if (noPrintTimesInMultiDay) {
							endTime = "";
						}
					} else {
						endTime = getStringForDate1(eDate);
					}
					if (printEndTime) {
						time = startTime + '-' + endTime;
						if (time == "-") {
							time = "";
						}
					} else {
						time = startTime;
					}
				} else {
					time = getStringForDate1(sDate);
				}
			} else if (sDate) {
				time = getStringForDate1(sDate);
			} else if (eDate) {
				time = getStringForDate1(eDate);
			}
	
			// Get calendar and category colours and apply them to the item's
			// table cell.
			let calColor = null;
			try {
				calColor = item.calendar.getProperty('color');
			} catch(ex) {
				let calMgr = Components.classes["@mozilla.org/calendar/manager;1"]
									.getService(Components.interfaces.calICalendarManager);
				calColor = calMgr.getCalendarPref(item.calendar, 'color'); 
			}
			if (!calColor) {
				calColor = "#A8C2E1";
			}
			//let pb2 = Components.classes["@mozilla.org/Services.prefs-service;1"]
            //                .getService(Components.interfaces.nsIPrefBranch2);
			let catColor;
			try {
				catColor = Services.prefs.getCharPref("calendar.category.color."+item.getProperty("CATEGORIES").toLowerCase());
			} catch(ex) {}
			if (!catColor) {
				for (let cat of item.getCategories({})) {
					try {
						catColor = Services.prefs.getCharPref("calendar.category.color." + cat.toLowerCase());
						break; // take first matching
					} catch(ex) {}
				}
			}
	
			let borderType = 'solid';
			if (item.status && printStatus) {
				if (item.status == "TENTATIVE") {
					borderType = 'dotted';
				}
				if (item.status == "CANCELLED") {
					borderType = 'dashed';
				}
			}
	
			let style = null;
			let tStyle = null;
			if (!oldcolour) {
				style = 'font-size: '+fsize+'px; border: ' + borderType + ' ' + calColor + ' 4px; ';
				style += ' color: black; ';
				tStyle = 'font-size: '+fsizet+'px; color: black;';
				if (catColor) {
					style += 'border-right-width: 8px; border-right-color: ' + catColor + ';';
					style += 'border-top-color: ' + catColor + ';';
					catColor = 0;
				}
			} else {
				style = 'font-size: '+fsize+'px; font-weight: bold; background-color: ' + calColor + ';';
				style += ' color: ' + getContrastingTextColor(calColor) + ';';
				tStyle = 'font-size: '+fsizet+'px; font-weight: bold; background-color: ' + calColor + '; color: ' + getContrastingTextColor(calColor) + ';';
				if (catColor) {
					style += ' border: solid ' + catColor + ' 2px;';
					catColor = 0;
				}
			}
			if (printPriority && item.priority && !oldcolour) {
				let prioCol = '';
				if (item.priority > 5) {
					prioCol = "#CFC";
				}
				if (item.priority == 5) {
					prioCol = "#FFC";
				}
				if (item.priority < 5) {
					prioCol = "#FCC";
				}
				style += 'background-color: ' + prioCol + ';';
			}
			let location = "";
			let descr = "";
			
			if (printLocation) {
				if (item.hasProperty('LOCATION')) {
					location = item.getProperty('LOCATION');
				}
			}
			if (printDescription) {
				if (item.hasProperty('DESCRIPTION')) {
					descr = item.getProperty('DESCRIPTION');
				}
			}
			let title = item.title;
			if (obeyPrivacy && (item.privacy == "PRIVATE" || item.privacy == "CONFIDENTIAL")) {
				title = "";
				descr = "";
				location = "";
			}
			
			// <tr width='100%'>
			//	<td valign='top' align='left' width='100%' style={style}>
			//	<span style={tStyle}>{time}</span> {title}
			//	<div class='desc'>{descr}</div>
			//	<div class='loc'>{location}</div></td>
			// </tr>; 
			let itemTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr"); 
			itemTr.setAttribute('width', '100%');
			let itemTd = document.createElementNS("http://www.w3.org/1999/xhtml","td"); 
			itemTd.setAttribute('valign', 'top');
			itemTd.setAttribute('align', 'left');
			itemTd.setAttribute('width', '100%');
			itemTd.setAttribute('style', style);
			itemTr.appendChild(itemTd);
			
			let itemTimeSpan = document.createElementNS("http://www.w3.org/1999/xhtml","span"); 
			itemTimeSpan.setAttribute('style', tStyle);
			itemTimeSpan.textContent = time;
			itemTd.appendChild(itemTimeSpan);
			itemTd.textContent += title;
			
			if(descr) {
				let itemDescDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div"); 
				itemDescDiv.className = 'desc';
				itemDescDiv.textContent = descr;
				itemTd.appendChild(itemDescDiv);
			}
			if(location) {
				let itemLocDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div"); 
				itemLocDiv.className = 'loc';
				itemLocDiv.textContent = location;
				itemTd.appendChild(itemLocDiv);
			}
			topTable.appendChild(itemTr);
		}
		// <tr height='100%'><td> </td></tr>
		let tmpTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr"); 
		tmpTr.setAttribute('height', '100%');
		let tmpTd =  document.createElementNS("http://www.w3.org/1999/xhtml","td"); 
		tmpTr.appendChild(tmpTd);
		topTable.appendChild(tmpTr);
		mainTd.appendChild(topTable);
		return mainTd;
	},

	/**
	 * Given a calIDateTime and an array of items, this function creates an HTML
	 * table containing the items, using the appropriate formatting and colours.
	 */
	getDayTd : function FGTimeSheetPrint_getDayTable(document, aDate, aItems, isOff) {
		// mainTd is the <td> element from the parent HTML table that will hold
		// the child HTML tables containing the date string and this day's items.
		let mainTd = document.createElementNS("http://www.w3.org/1999/xhtml","td"); //<td border='0' width="100%" valign='top'/>
		mainTd.setAttribute('border', '0');
		mainTd.setAttribute('width', '100%');
		mainTd.setAttribute('valign', 'top');
		let dateFormatter = Components.classes["@mozilla.org/calendar/datetime-formatter;1"]
                               .getService(Components.interfaces.calIDateTimeFormatter);

		let printAllDay = Services.prefs.getBoolPref("fgprinters.ts.printallday", false);
		let printLocation = Services.prefs.getBoolPref("fgprinters.ts.printlocation", true);
		let printDescription = Services.prefs.getBoolPref("fgprinters.ts.printdescription", true);
		let printPriority = Services.prefs.getBoolPref("fgprinters.ts.printpriority", true);
		let printStatus = Services.prefs.getBoolPref("fgprinters.ts.printstatus", true);
		let printEndTime = Services.prefs.getBoolPref("fgprinters.ts.printendtime", true);
		let obeyPrivacy = Services.prefs.getBoolPref("fgprinters.obeyprivacy", true);
		let startDay = Services.prefs.getIntPref("fgprinters.ts.starthour", 7);
		let displayHours = Services.prefs.getIntPref("fgprinters.ts.displayhours", 15);
		let endDay = startDay + displayHours;
		let noPrintTimesInMultiDay = Services.prefs.getBoolPref("fgprinters.ts.noprinttimesinmultidayevents", false);
		
		if (endDay > 24) {
			Components.utils.reportError("Too many hours to display (" + endDay + "), adjusting...");
			endDay = 24;
			displayHours = endDay - startDay;
		}
		let defaultTimezone = null;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {}
		let fsize = Services.prefs.getIntPref("fgprinters.ts.fontsize", 12);
		let fsizet = Services.prefs.getIntPref("fgprinters.ts.fontsizetime", 12);
		let oldcolour = Services.prefs.getBoolPref("fgprinters.ts.oldcolour", false);

        function getStringForDate(aDate) {
            if (!aDate.isDate) {
                return dateFormatter.formatTime(aDate);
            }

            // We cache the string for "All Day" 'cause we're good like that.
            if (this.mAllDayString == null) {
                let sbs = Components.classes["@mozilla.org/intl/stringbundle;1"]
                                    .getService(Components.interfaces.nsIStringBundleService);
                let props = sbs.createBundle("chrome://calendar/locale/dateFormat.properties");
                this.mAllDayString = props.GetStringFromName("AllDay");
            }
            if (printAllDay) { return this.mAllDayString; } else { return ""; }
        }


        function isEventInRange(item) {
            let sDate = item.startDate || item.entryDate || item.dueDate;
            let eDate = item.endDate || item.dueDate || item.entryDate;
            if ((sDate.compare(aDate) <= 0) && (eDate.compare(aDate) >= 0)) {
                return 1;
            }
        }

		let filteredEvents = aItems.filter(isEventInRange);

        function isNonAllDay(item) {
            let sDate = item.startDate || item.entryDate || item.dueDate;
            let eDate = item.endDate || item.dueDate || item.entryDate;
            if ((sDate && !sDate.isDate) && (eDate && !eDate.isDate)) {
                return 1;
            }
        }

		let filteredNonAllDayEvents = filteredEvents.filter(isNonAllDay);
		let alreadyPrinted = Array(filteredNonAllDayEvents.length);
		for (let eventIndex = 0; eventIndex < filteredNonAllDayEvents.length; eventIndex++) {
			alreadyPrinted[eventIndex] = 0;
		}
		// <table width='100%' valign='top' class="weekend" style='margin-top: -4px; margin-left: -1px; margin-right: -2px;'><tr height='500px'/></table>
		let innerTable = document.createElementNS("http://www.w3.org/1999/xhtml","table");
		innerTable.setAttribute('width', '100%');
		innerTable.setAttribute('valign', 'top');
		innerTable.setAttribute('style', 'margin-top: -4px; margin-left: -2px; margin-right: -2px;');
		if (isOff) { 
			innerTable.className = 'weekend'; 
			innerTable.setAttribute('style', 'margin-top: -4px; margin-bottom: -4px; margin-left: -1px; margin-right: -2px;');
		}
		let innerTableTr = document.createElementNS("http://www.w3.org/1999/xhtml","tr");
		innerTableTr.setAttribute('height', '500px');
		innerTable.appendChild(innerTableTr);
		if (!filteredNonAllDayEvents.length) {
			let tmpTd = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			//<td valign='top' align='left' width='100%' height='100%'/>
			tmpTd.setAttribute('valign', 'top');
			tmpTd.setAttribute('align', 'left');
			tmpTd.setAttribute('width', '100%');
			tmpTd.setAttribute('height', '100%');
			innerTable.appendChild(tmpTd);
		}
	
		for (let eventIndex = 0; eventIndex < filteredNonAllDayEvents.length; eventIndex++) {
			// don't bother if we already printed this one
			if (alreadyPrinted[eventIndex]) {
				continue;
			}
			alreadyPrinted[eventIndex] = 1;
			let item = filteredNonAllDayEvents[eventIndex];
			let sDate = item.startDate || item.entryDate || item.dueDate;
			let eDate = item.endDate || item.dueDate || item.entryDate;
			if (sDate) {
				try {
					sDate = sDate.getInTimezone(defaultTimezone);
				} catch(ex) {}
			} 
			if (eDate) {
				try {
					eDate = eDate.getInTimezone(defaultTimezone);
				} catch(ex) {}
			}

			// End dates are exclusive. Adjust the eDate accordingly.
			if (sDate && sDate.isDate && eDate) {
				eDate = eDate.clone();
				eDate.day -= 1;
				try {
					eDate.normalize();
				} catch(ex) {}
			}

			// If the item has no end date, or if the item's end date is aDate or
			// is before aDate, skip to the next item.
			if (!eDate || (eDate.compare(aDate) < 0)) {
				continue;
			}

			// No start date or a start date that's after the date we want is bad.
			if (!sDate || (sDate.compare(aDate) > 0)) {
				break;
			}
	
			//<td valign='top' align='left' width='100%'/>;
			let printItem = document.createElementNS("http://www.w3.org/1999/xhtml","td");
			printItem.setAttribute('valign', 'top');
			printItem.setAttribute('align', 'left');
			printItem.setAttribute('width', '100%');
			function addItemToTd(event, prev) {
				let sDate = event.startDate || event.entryDate || event.dueDate;
				let eDate = event.endDate || event.dueDate || event.entryDate;
				if (sDate) {
					try {
						sDate = sDate.getInTimezone(defaultTimezone);
					} catch(ex) {}
				}
				if (eDate) {
					try {
						eDate = eDate.getInTimezone(defaultTimezone);
					} catch(ex) {}
				}
				let subPrev = 0;
		
				// End dates are exclusive. Adjust the eDate accordingly.
				if (sDate && sDate.isDate && eDate) {
					eDate = eDate.clone();
					eDate.day -= 1;
					try {
						eDate.normalize();
					} catch(ex) {}
				}
				let time;
				if (sDate && eDate) {
					if (!sDate.isDate) {
						let startTime = "";
						let endTime = "";
						if (sDate.compare(aDate) < 0) {
							let dummyDate = sDate.clone();
							dummyDate.hour = 0;
							dummyDate.minute = 0;
							try {
								dummyDate.normalize();
							} catch(ex) {}
							startTime = getStringForDate(dummyDate);
							if (noPrintTimesInMultiDay) {
								startTime = "";
							}
						} else {
							startTime = getStringForDate(sDate);
						}
						if (eDate.compare(aDate) > 0) {
							let dummyDate = eDate.clone();
							dummyDate.hour = 23;
							dummyDate.minute = 59;
							try {
								dummyDate.normalize();
							} catch(ex) {}
							endTime = getStringForDate(dummyDate);
							if (noPrintTimesInMultiDay) {
								endTime = "";
							}
						} else {
							endTime = getStringForDate(eDate);
						}
						if (printEndTime) {
							time = startTime + '-' + endTime;
							if (time == "-") {
								time = "";
							}
						} else {
							time = startTime;
						}
					} else {
						time = getStringForDate(sDate);
					}
				} else if (sDate) {
					time = getStringForDate(sDate);
				} else if (eDate) {
					time = getStringForDate(eDate);
				}
		
				// Get calendar and category colours and apply them to the item's
				// table cell.
				let calColor = null;
				try {
					calColor = event.calendar.getProperty('color');
				} catch(ex) {
					let calMgr = Components.classes["@mozilla.org/calendar/manager;1"]
										.getService(Components.interfaces.calICalendarManager);
					calColor = calMgr.getCalendarPref(event.calendar, 'color'); 
				}
				if (!calColor) {
					calColor = "#A8C2E1";
				}
				//let pb2 = Components.classes["@mozilla.org/Services.prefs-service;1"]
				//					.getService(Components.interfaces.nsIPrefBranch2);
				let catColor;
				try {
					catColor = Services.prefs.getChar("calendar.category.color."+event.getProperty("CATEGORIES").toLowerCase());
				} catch(ex) {}
				if (!catColor) {
					for (let cat of event.getCategories({})) {
						try {
							catColor = Services.prefs.getCharPref("calendar.category.color." + cat.toLowerCase());
							break; // take first matching
						} catch(ex) {}
					}
				}
				
				let begin = sDate.subtractDate(aDate).inSeconds / 60;
				let end = eDate.subtractDate(aDate).inSeconds / 60;
				if (begin < startDay * 60) { begin = startDay * 60; }
				if (begin > endDay * 60) { begin = endDay * 60; }
				if (end > endDay * 60) { end = endDay * 60; }
				if (end < startDay * 60) { end = startDay * 60; }
				begin -= startDay * 60;
				end -= startDay * 60;
				if (end < begin) {
					end = begin;
				}
				let eDuration = Math.round((end - begin) * 500 / (displayHours * 60)) - 4; // double height of border
				if (eDuration < 0) {
					eDuration = 2;
				}
				let borderType = 'solid';
				if (event.status && printStatus) {
					if (event.status == "TENTATIVE") {
						borderType = 'dotted';
					}
					if (event.status == "CANCELLED") {
						borderType = 'dashed';
					}
				}
				
				let eStart = Math.round(begin * 500 / (displayHours * 60)) - prev;
				let style = null;
				let tStyle = null;
				if (!oldcolour) {
					style = 'font-size: '+fsize+'px; ';
					tStyle = 'font-size: '+fsizet+'px; color: black;';
					if (eDuration) {
						style += 'border: ' + borderType + ' ' + calColor + ' 2px; ';
					}
					style += ' color: black; ';
					if (catColor && eDuration) {
						style += 'border-right-width: 4px; border-right-color: ' + catColor + ';';
						style += 'border-top-color: ' + catColor + ';';
						catColor = 0;
					}
				} else {
					style = 'font-size: '+fsize+'px; font-weight: bold; background-color: ' + calColor + ';';
					style += ' color: ' + getContrastingTextColor(calColor) + ';';
					tStyle = 'font-size: '+fsizet+'px; font-weight: bold; background-color: ' + calColor + '; color: ' + getContrastingTextColor(calColor) + ';';
					if (catColor) {
						style += ' border: solid ' + catColor + ' 2px;';
						catColor = 0;
					}
				}
				style += 'margin-top: ' + eStart + 'px; height: ' + eDuration + 'px; margin-left: -2px; margin-right: -2px; ';
				if (printPriority && event.priority && !oldcolour) {
					let prioCol = '';
					if (event.priority > 5) {
						prioCol = "#CFC";
					}
					if (event.priority == 5) {
						prioCol = "#FFC";
					}
					if (event.priority < 5) {
						prioCol = "#FCC";
					}
					style += 'background-color: ' + prioCol + ';';
				}
				let location = "";
				let descr = "";
				
				if (printLocation) {
					if (event.hasProperty('LOCATION')) {
						location = event.getProperty('LOCATION');
					}
				}
				if (printDescription) {
					if (event.hasProperty('DESCRIPTION')) {
						descr = event.getProperty('DESCRIPTION');
					}
				}
				let title = event.title;
				if (obeyPrivacy && (event.privacy == "PRIVATE" || event.privacy == "CONFIDENTIAL")) {
					title = "";
					descr = "";
					location = "";
				}
				
				//<div style={style}><span style={tStyle}>{time}</span> {title}				
				let itemTimeTitleDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div"); 
				itemTimeTitleDiv.setAttribute('style', style);
				let itemTimeSpan = document.createElementNS("http://www.w3.org/1999/xhtml","span"); 
				itemTimeSpan.setAttribute('style', tStyle);
				itemTimeSpan.textContent = time;
				itemTimeTitleDiv.appendChild(itemTimeSpan);
				printItem.appendChild(itemTimeTitleDiv);
				let itemTitleSpan = document.createElementNS("http://www.w3.org/1999/xhtml","span"); 
				itemTitleSpan.textContent = ' ' + title;
				itemTimeTitleDiv.appendChild(itemTitleSpan);
			
				if(descr) {
					let itemDescDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div"); 
					itemDescDiv.className = 'desc';
					itemDescDiv.textContent = descr;
					itemTimeTitleDiv.appendChild(itemDescDiv);
				}
				if(location) {
					let itemLocDiv = document.createElementNS("http://www.w3.org/1999/xhtml","div"); 
					itemLocDiv.className = 'loc';
					itemLocDiv.textContent = location;
					itemTimeTitleDiv.appendChild(itemLocDiv);
				}
				
				return eStart+eDuration+prev +4; // width of bottom border
			}
			
			subPrev = addItemToTd(item, 0);
			let compareDate = eDate.clone();
			
			for (let newIndex = eventIndex; newIndex < filteredNonAllDayEvents.length; newIndex++) {
				if (alreadyPrinted[newIndex]) {
					continue;
				}
				let item = filteredNonAllDayEvents[newIndex];
				let sDate = item.startDate || item.entryDate || item.dueDate;
				let eDate = item.endDate || item.dueDate || item.entryDate;
				if (sDate) {
					try {
						sDate = sDate.getInTimezone(defaultTimezone);
					} catch(ex) {}
				}
				if (eDate) {
					try {
						eDate = eDate.getInTimezone(defaultTimezone);
					} catch(ex) {}
				}
	
				// End dates are exclusive. Adjust the eDate accordingly.
				if (sDate && sDate.isDate && eDate) {
					eDate = eDate.clone();
					eDate.day -= 1;
					try {
						eDate.normalize();
					} catch(ex) {}
				}
				if (compareDate.compare(sDate) <= 0) {
					alreadyPrinted[newIndex] = 1;
					subPrev = addItemToTd(item, subPrev);
					compareDate = eDate.clone();
				}
			}
			innerTableTr.appendChild(printItem);
		}
		mainTd.appendChild(innerTable);
		return mainTd;
	},
	
	setupStyleSheet: function FGlist_setupStyleSheet(html) {
		//
		let fsizeday = Services.prefs.getIntPref("fgprinters.ts.fontsizedayname", 20);
		let fsizetitle = Services.prefs.getIntPref("fgprinters.ts.fontsizetitle", 20);
		let dnbk = Services.prefs.getCharPref("fgprinters.headcolour", "#f0f0f0");
		let wec = Services.prefs.getCharPref("fgprinters.weekendcolour", "#FFFFEE");
		let sansserif = Services.prefs.getBoolPref("fgprinters.sansserif", true);
		let fsizeloc = Services.prefs.getIntPref("fgprinters.ts.fontsizelocation", 14);
		let fsizedesc = Services.prefs.getIntPref("fgprinters.ts.fontsizedescription", 14);
		let printLocationRight = Services.prefs.getBoolPref("fgprinters.ts.printlocationright", true);
		let fsizeglobaltitle = Services.prefs.getIntPref("fgprinters.fontsizetitle", 26);
		let fsizesubtitle = Services.prefs.getIntPref("fgprinters.fontsizesubtitle", 20);
		
		let style = ".main-table { font-size: "+fsizetitle+"px; font-weight:bold; border:solid 1px black; border-collapse:collapse;}\n";
		style += ".main-title { font-size: "+fsizeglobaltitle+"px; font-weight:bold; text-align: center; }\n";
		style += ".sub-title { font-size: "+fsizesubtitle+"px; font-weight:bold; text-align: center; }\n";
		style += ".day-name { border: 1px solid #000; background-color: "+dnbk+"; font-size: "+fsizeday+"px; font-weight: bold; }\n";
		if (sansserif) { style += "body, tr, td, table, p { font-family: Arial, sans-serif; }\n"; }
		style += ".weekend { background-color: "+wec+"; }\n";
		if (printLocationRight) {
			style += ".loc { font-size: "+fsizeloc+"px; text-align: right; }\n";
		} else {
			style += ".loc { font-size: "+fsizeloc+"px; text-align: left; }\n";
		}
		style += ".desc { font-size: "+fsizedesc+"px; text-align: left; }\n";	
		html.getElementsByTagName('style')[0].textContent += "\n" + style;
	}
};