/* -*- Mode: javascript; tab-width: 20; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Calendar code.
 *
 * The Initial Developer of the Original Code is
 *   Joey Minta <jminta@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * The output code for the time sheet was written by
 *   Ferdinand Grassmann <ferdinand@grassmann.info>
 *
 * Contributor(s):
 *   Matthew Willis <lilmatt@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

var { Services } = ChromeUtils.import("resource://gre/modules/Services.jsm");
var { cal } = ChromeUtils.import("resource://calendar/modules/calUtils.jsm");
 
Components.utils.import("resource://calendar/modules/calUtils.jsm");
 
/**
 * Prints a time sheet view of a week of events
 */

function NGcalDayPrinter() {
}

var NGcalDayPrinterID = Components.ID('{9083C5EC-5213-11DC-BF8C-418855D89595}');
var NGcalDayPrinterInterfaces = [Components.interfaces.calIPrintFormatter];
var NGcalDayPrinterContractID = "@mozilla.org/calendar/printformatter;1?type=ng-dayprinter";
var NGcalDayPrinterDescription = "Calendar Daily TimeSheet Print Formatter";

NGcalDayPrinter.prototype = {
    classID: NGcalDayPrinterID,
	QueryInterface: ChromeUtils.generateQI([Ci.calIPrintFormatter]),
	
	
 

    get name() {
        return "NG - " + cal.l10n.getAnyString("fgprinters", "fgprinters", "daysheet", []);
    },

	formatToHtml : function dayPrinter_format(aStream, aStart, aEnd, aCount, aItems, aTitle) {
		// FIXME: Should not be necessary, but it is in fact...
		if(typeof(cal.xml) == "undefined") {
	        Components.utils.import("resource://fgprinters/modules/calXMLUtils.jsm");
		}
		
		// Load the HTML template file  FIXME: might be useful to allow custom template
		let document = cal.xml.parseFile("chrome://fgprinters/skin/printing/calDayPrinter.html");
		
		// Initialize title and header
		this.setupHeader(document, aStart, aEnd, aCount, aItems, aTitle);
		
		// Prepare and clean events list
		let [start, end, sortedList] = this.filterAndSortItems(aItems, aStart, aEnd);
		
		this.setupLegend(document, sortedList);
		
		let date = start.clone();
		while(date.compare(end) == -1) {
			// Process week
			this.setupDay(document, date, sortedList);
			// Next week
			date.day += 1;
		}
		
		// Add tasks with no due date
		this.setupTaskWithNoDate(document, aStart, aEnd, aCount, aItems, aTitle);
		
		// Add custom stylesheet if any :
		this.setupStyleSheet(document, aStart, aEnd, aCount, aItems, aTitle);
		
		// Remove templates from HTML, no longer needed
        let templates = document.getElementById("templates");
        templates.parentNode.removeChild(templates);	
		
		// Stream out the resulting HTML
        let html = cal.xml.serializeDOM(document);
		//Components.utils.reportError(html);
        let convStream = Components.classes["@mozilla.org/intl/converter-output-stream;1"]
                                   .createInstance(Components.interfaces.nsIConverterOutputStream);
        convStream.init(aStream, 'UTF-8', 0, 0x0000);
        convStream.writeString(html);
	},
	
	filterAndSortItems: function dayPrinter_filterAndSortItems(aItems, aStart, aEnd) {
		// helper: returns the passed item's startDate, entryDate or dueDate, in
		//         that order. If the item doesn't have one of those dates, this
		//         doesn't return.
		function hasUsableDate(item) {
			return item.startDate || item.entryDate || item.dueDate;
		}
		
		// Clean out the item list so it only contains items we will want to
		// include in the printout.
		let filteredItems = aItems.filter(hasUsableDate);
		let sortedList = filteredItems.sort(cal.print.comparePrintItems);
		
		// Start at the beginning of the week that aStart is in, and loop until
		// we're at aEnd. In the loop we build the HTML table for each day, and
		// get the day's items using getDayTd().
		let start = aStart || sortedList[0].startDate || sortedList[0].entryDate ||
					sortList[0].dueDate;
		cal.ASSERT(start, "can't find a good starting date to print");
	
		let lastItem = sortedList[sortedList.length-1];
		let end = aEnd || lastItem.startDate || lastItem.entryDate ||
				lastItem.dueDate;
		cal.ASSERT(end, "can't find a good ending date to print");
		return [start, end, sortedList];
	},
	
	/**
	 *  Setup the title, etc.
	 */
	setupHeader: function dayPrinter_setupHeader(document, aStart, aEnd, aCount, aItems, aTitle) {
		// Set page title
        document.getElementById("title").textContent = aTitle;
		
		// Title and Subtitle
        let printtitle = Services.prefs.getBoolPref("fgprinters.printtitle", false);
        let subTitle = Services.prefs.getCharPref("fgprinters.subtitle", "");
        
		if (printtitle && aTitle && (aTitle != "Untitled")) {
            document.getElementById('mainTitle').textContent = aTitle;
			document.getElementById('mainTitle').style.display = '';
        } else {
		    document.getElementById('mainTitle').style.display = 'none';
		}
        if (subTitle) {
            document.getElementById('subTitle').textContent = subTitle;
			document.getElementById('subTitle').style.display = '';
        } else {
		    document.getElementById('subTitle').style.display = 'none';
		}
	},
	
	setupLegend: function dayPrinter_setupLegend(document, aItems) {
		let processedCalendars = {};
		let calendars = [];
		
		aItems.forEach( function(item) {
		   if(! processedCalendars[item.calendar.id]) {
				processedCalendars[item.calendar.id] = true;
				calendars.push( { name: item.calendar.name,  id: cal.view.formatStringForCSSRule(item.calendar.id)} );
		   }
		});
		
		for(let i=0; i < calendars.length; i++) {
			let spanNode = document.getElementById('templates.legendItem').cloneNode(true);
			spanNode.removeAttribute('id');
			spanNode.querySelector('.calendar-color-box').setAttribute('calendar-id', calendars[i].id);
			spanNode.querySelector('.calendar-color-box').textContent = " ";
			spanNode.querySelector('.calendar-name').textContent = calendars[i].name;
			while(spanNode.firstChild) {
				document.getElementById('legend').appendChild(spanNode.firstChild);
			}
		}
	},
	
	setupDay: function dayPrinter_setupDay(document, date, aItems) {
		let dayTable = document.getElementById('templates.day').cloneNode(true);
		// Day id
		dayTable.setAttribute("id", 'day-' + date.year + '-' + date.month + '-' + date.day);

		let tmpDate = date.clone();
		// Day id
		let days = dayTable.querySelectorAll('.day0');
		for(let d=0; d < days.length; d++) {
			days[d].setAttribute('id', 'day-' + tmpDate.year +'-'+tmpDate.month + '-'+tmpDate.day);
		}
		// Day title
		dayTable.querySelector('.table-headers')
				 .querySelector('.day0').textContent = this.getDateString(tmpDate);
		// working days
		let isOff = this.isOffline(tmpDate.weekday);
		if(isOff) {
			let days = dayTable.querySelectorAll('.day-table' )
			for(let d=0; d < days.length; d++) {
				days[d].className += ' weekend';
			}
		}
		
		// Set offline hours
		let startHour = Services.prefs.getIntPref('calendar.view.daystarthour', 0);
		for(let h=0; h < startHour; h++) {
			dayTable.querySelector('.h-'+h+'-00').className += ' offline';
		}
		let endHour = Services.prefs.getIntPref('calendar.view.dayendhour', 23);
		for(let h=endHour; h < 24; h++) {
			dayTable.querySelector('.h-'+h+'-00').className += ' offline';
		}

		// Fill Allday events for this day
		this.setupAllDayEvents(document, dayTable, tmpDate, aItems);
			
		// Fill non-allday events for this day
		let showNonWorkingHours = Services.prefs.getBoolPref("calendar.printing.day.shownonworkinghours", true);
		if(showNonWorkingHours) {
			this.setupDayEvents(document, dayTable, tmpDate, aItems, 0, 23);
		} else {
			this.setupDayEvents(document, dayTable, tmpDate, aItems, startHour, endHour);
		}
		
		// Add the week to the page
		while(dayTable.firstChild) {
			document.getElementById('days-container').appendChild(dayTable.firstChild);
		}
	},	
	
	/**
	 *  Add allday items to current day
	 */
	setupAllDayEvents : function dayPrinter_setupAllDayEvents(document, dayTable, aDate, aItems) {
		
		let me = this;
		
		function isEventInRange1(item) {
            let sDate = item.startDate || item.entryDate || item.dueDate;
            let eDate = item.endDate || item.dueDate || item.entryDate;
            if (sDate && (sDate.compare(aDate) <= 0) && (eDate.compare(aDate) > 0)) {
                return 1;
            }
        }
		let filteredEvents = aItems.filter(isEventInRange1);
		function isAllDay(item) {
           let sDate = item.startDate || item.entryDate || item.dueDate;
		   if(sDate.isDate) {
				return 1;
		   }
        }
		filteredEvents = filteredEvents.filter(isAllDay);
		for (let item of filteredEvents) {
			if(!item) continue;
			let eventNode = document.getElementById('templates.allDayEvent').cloneNode(true);
			eventNode.removeAttribute('id');
			eventNode.querySelector('.calendar-color-box').setAttribute("calendar-id", cal.view.formatStringForCSSRule(item.calendar.id));
			me.insertCalendarRules(document, item.calendar);
			eventNode.querySelector('.event-title').textContent = item.title;
			if(typeof(item.description) != 'undefined') {
				eventNode.querySelector('.event-description').textContent = item.description;
			}
			try { 
				if(item.hasProperty('LOCATION')) {
					eventNode.querySelector('.event-location').textContent = item.getProperty('LOCATION');
				}
			} catch(e){}
			if (item.priority > 5) {
					eventNode.querySelector('.calendar-color-box').className += ' priority-high';
            }
            if (item.priority == 5) {
				eventNode.querySelector('.calendar-color-box').className += ' priority-medium';
            }
            if (item.priority < 5) {
                eventNode.querySelector('.calendar-color-box').className += ' priority-low';
            }
			let dayContainer = dayTable.querySelector('.all-day').querySelector('#day-' + aDate.year + '-' + aDate.month + '-' + aDate.day);
			dayContainer = dayContainer.querySelector('.day-container');
			if(dayContainer) {
				while(eventNode.firstChild) {
					dayContainer.appendChild(eventNode.firstChild);
				}
			}
		}
	},
	
	/**
	 *  Layout non-allday events 
	 *
	 */
	setupDayEvents : function dayPrinter_setupDayEvents(document, dayTable, aDate, aItems, startHour, endHour) {
		
		let me = this;
		
		let totalHeight = Services.prefs.getIntPref("calendar.printing.day.hourtableheight", 400);
		// Compute rowHeight		
		let displayedRows = endHour - startHour;
		let rowHeight = totalHeight / displayedRows;
		
		// Set rows height
		let rowsGroup = dayTable.querySelector('.hours-rows');  // Might not be necessary, simplify
		rowsGroup.style.height = totalHeight + 'px';
		let rows = dayTable.querySelectorAll('.hour');
		for(let i=0; i < rows.length; i++) {
			rows[i].style.height = rowHeight + 'px';
		}
		// Start of day
		let dayStart = aDate.clone(); dayStart.hour = 0; dayStart.minute = 0;
		// End of day
		let dayEnd = aDate.clone(); dayEnd.hour = 23; dayEnd.minute = 59;
		function isEventInRange1(item) {
            let sDate = item.startDate || item.entryDate || item.dueDate;
            let eDate = item.endDate || item.dueDate || item.entryDate;

            if (sDate && eDate && (sDate.compare(dayStart) <= 0) && (eDate.compare(dayStart) >= 0)) {
                return 1;
            } 
			if (sDate && eDate && (sDate.compare(dayEnd) <= 0) && (eDate.compare(dayEnd) >= 0)) {
                return 1;
            }
			if (sDate && eDate && (sDate.compare(dayStart) >= 0) && (eDate.compare(dayEnd) <= 0)) {
                return 1;
            }
        }
		let filteredEvents = aItems.filter(isEventInRange1);
		function isNotAllDay(item) {
           let sDate = item.startDate || item.entryDate || item.dueDate;
		   if(!sDate.isDate) {
				return 1;
		   }
        }
		filteredEvents = filteredEvents.filter(isNotAllDay);
		let defaultTimezone = cal.dtz.defaultTimezone;
		let eventMap = this.computeEventMap(document, filteredEvents, aDate, startHour, endHour);

		eventMap.forEach(function(blobInfo) {
			// {blob: currentBlob, totalCols: colEndArray.length}
			let totalCols = blobInfo.totalCols;

			let currentBlob = blobInfo.blob;
			currentBlob.forEach(function(blobItemInfo) {
				// {itemInfo: curItemInfo, startCol: colEndArray.length, colSpan: 1}
				let itemInfo = blobItemInfo.itemInfo;
				let startCol = blobItemInfo.startCol;
				let colSpan = blobItemInfo.colSpan
				let item = itemInfo.event;
				// Calculate position and height
				let begin = itemInfo.layoutStart;
				let end =  itemInfo.layoutEnd;
				let offset = begin.minute / 60 * rowHeight;
				let height = end.subtractDate(begin).inSeconds / 3600 * rowHeight;
				// Item start-end
				let beginDate = item.startDate || item.entryDate || item.dueDate;
				let endDate = item.endDate|| item.entryDate || item.dueDate;
				
				let eventNode = document.getElementById('templates.dayEvent').cloneNode(true);
				eventNode.removeAttribute('id');
				eventNode.querySelector('.calendar-color-box').setAttribute("calendar-id", cal.view.formatStringForCSSRule(item.calendar.id));
				me.insertCalendarRules(document, item.calendar);
				let sInterval = beginDate.hour + ':' + beginDate.minute + '-' + endDate.hour  +':' + endDate.minute;
				eventNode.querySelector('.event-interval').textContent = sInterval;
				eventNode.querySelector('.event-title').textContent = item.title;
				if(typeof(item.description) != 'undefined') {
					eventNode.querySelector('.event-description').textContent = item.description;
				}
				
				try { 
					if(item.hasProperty('LOCATION')) {
						eventNode.querySelector('.event-location').textContent = item.getProperty('LOCATION');
					}
				} catch(e){}
				if (item.priority > 5) {
					eventNode.querySelector('.calendar-color-box').className += ' priority-high';
				}
				if (item.priority == 5) {
					eventNode.querySelector('.calendar-color-box').className += ' priority-medium';
				}
				if (item.priority < 5) {
					eventNode.querySelector('.calendar-color-box').className += ' priority-low';
				}
				eventNode.querySelector('.event-container').style.height = height + 'px';
				eventNode.querySelector('.event-container').style.top = offset + 'px';
				eventNode.querySelector('.day-events-col').style.width = (colSpan / totalCols * 100) + '%';
				eventNode.querySelector('.day-events-col').setAttribute('colspan', colSpan);
				let hour = '.h-'+begin.hour+'-00';
				let dayId = '#day-' + aDate.year + '-' + aDate.month + '-' + aDate.day;
				let dayContainer = dayTable.querySelector(hour).querySelector(dayId).querySelector('.day-events-row');
				
				// Add spacers before if needed
				for(let c = dayContainer.children.length; c <  startCol; c++) {
					// Add empty item
					let spacerNode = document.getElementById('templates.dayEvent').cloneNode(true);
					//spacerNode.removeAttribute('id');
					spacerNode.querySelector('.day-events-col').setAttribute('colspan', 1);
					spacerNode.querySelector('.day-events-col').className += ' event-spacer';
					while(spacerNode.lastChild) {
						dayContainer.insertBefore(spacerNode.lastChild, dayContainer.firstChild);
					}
				}
				// Add the event
				if(dayContainer) {
					while(eventNode.firstChild) {
						dayContainer.appendChild(eventNode.firstChild);
					}
				} else {
					Components.utils.reportError('Cannot add event '+item.title +' to day: '+hour + ' of day ' +dayId);
				}
			});
		});
		
		eventMap.forEach(function(blobInfo) {
			let currentBlob = blobInfo.blob;
			let totalCols = blobInfo.totalCols;
			currentBlob.forEach(function(blobItemInfo) {
				// {itemInfo: curItemInfo, startCol: colEndArray.length, colSpan: 1}
				let item = blobItemInfo.itemInfo.event;
				let begin = blobItemInfo.itemInfo.layoutStart;
				begin = begin.getInTimezone(defaultTimezone);				
				let hour = '.h-'+begin.hour+'-00';
				let dayId = '#day-' + aDate.year + '-' + aDate.month + '-' + aDate.day;
				let dayContainer = dayTable.querySelector(hour).querySelector(dayId).querySelector('.day-events-row');

				let totalColSpan = 0;
				for(let i=0; i < dayContainer.children.length; i++){
					totalColSpan += parseInt(dayContainer.children[i].getAttribute("colspan"));
				}
				// Add spacers after if needed
				for(let c = (totalColSpan); c < totalCols; c++) {
					// Add empty item
					let spacerNode = document.getElementById('templates.dayEvent').cloneNode(true);
					//spacerNode.removeAttribute('id');
					spacerNode.querySelector('.day-events-col').setAttribute('colspan', 1);
					spacerNode.querySelector('.day-events-col').className += ' event-spacer';
					while(spacerNode.firstChild) {
						dayContainer.appendChild(spacerNode.firstChild);
					}
				}
			});
		});
	},

	setupTaskWithNoDate: function setupTaskWithNoDate(document, aStart, aEnd, aCount, aItems, aTitle) {
        
		let taskContainer = document.getElementById("task-container");

        let taskListBox = document.getElementById("tasks-list-box");
        if (taskListBox.hasAttribute("hidden")) {
            let tasksTitle = document.getElementById("tasks-title");
            taskListBox.removeAttribute("hidden");
            tasksTitle.textContent = cal.l10n.getCalString("tasksWithNoDueDate");
        }
		
		function isTaskWithNoUsableDate(item) {
			if(cal.item.isToDo(item) && !(item.startDate || item.entryDate || item.dueDate)) {
			    return 1;
			}
		}
		let filteredItems = aItems.filter(isTaskWithNoUsableDate);
		
		if(filteredItems.length == 0) {
		    taskListBox.setAttribute("hidden", "true");
		}
		
		filteredItems.forEach(function(item) {
			let taskNode = document.getElementById("templates.task").cloneNode(true);
			taskNode.removeAttribute("id");
			// Fill in details of the task
			if (item.isCompleted) {
				taskNode.querySelector(".task-checkbox").setAttribute("checked", "checked");
			}
			taskNode.querySelector(".task-title").textContent = item.title;
	
			while(taskNode.firstChild) {
				taskContainer.appendChild(taskNode.firstChild);
			}
		});
    },
	
	setupStyleSheet: function timeSheet_setupStyleSheet(document) {
		// Add custom styles
		let style = '';
		let showNonworkingHours = Services.prefs.getBoolPref('calendar.printing.day.shownonworkinghours', false);
		if(!showNonworkingHours) {
		  style += '.offline { display:none; }\n';
		}
		
		let printInterval = Services.prefs.getBoolPref('calendar.printing.day.printinterval', false);
		if(!printInterval) {
			style += '.event-interval { display:none; }\n';
		}
		
		
		let printDayOff = Services.prefs.getBoolPref('calendar.printing.day.showdayoff', true);
		if(!printDayOff) {
			style += '.weekend { display:none; }\n';
		}
		
		document.getElementById('sheet').textContent += "\n" + style;
		
		// Add user's stylesheet if any
		// e.g: chrome://calendar-custom-printing/skin/printing/calTimeSheetPrinter.css
		// FIXME: allow muliple files, separator would be comma
		let customStyleSheet = Services.prefs.getBoolPref('calendar.custom.printing.day.stylesheet', false);
		if(customStyleSheet) {
			let cssNode = document.createElement('link');
			cssNode.setAttribute('rel', 'stylesheet');
			cssNode.setAttribute('type', 'text/css');
			cssNode.setAttribute('media', 'all');
			cssNode.setAttribute('href', customStyleSheet);
			try {
				document.getElementsByTagName('head')[0].appendChild(cssNode);
			} catch(e) {
			}
		}
	},

	/*   ------- THIS CODE COMES FROM LIGHTNING CORE --------------------   
	 * We're going to create a series of 'blobs'.  A blob is a series of
	 * events that create a continuous block of busy time.  In other
	 * words, a blob ends when there is some time such that no events
	 * occupy that time.
	 *
	 * Each blob will be an array of objects with the following properties:
	 *    item:     the event/task
	 *    startCol: the starting column to display the event in (0-indexed)
	 *    colSpan:  the number of columns the item spans
	 *
	 * An item with no conflicts will have startCol: 0 and colSpan: 1.
	 */
	computeEventMap: function computeEventMap(document, aItems, aDate, startHour, endHour) {
		let defaultTimezone = cal.dtz.defaultTimezone;
		// Start of day
		let dayStart = aDate.clone().getInTimezone(defaultTimezone);
		dayStart.isDate = 0; dayStart.hour = 0; dayStart.minute = 0;
		let workingDayStart = aDate.clone().getInTimezone(defaultTimezone); 
		workingDayStart.isDate = 0; workingDayStart.hour = startHour; workingDayStart.minute = 0;
	
		// End of day
		let dayEnd = aDate.clone(); 
		dayEnd = dayEnd.getInTimezone(defaultTimezone);
		dayEnd.isDate = 0; dayEnd.hour = 23; dayEnd.minute = 59;
		let workingDayEnd = aDate.clone(); 
		workingDayEnd.isDate = 0; workingDayEnd.hour = endHour; workingDayEnd.minute = 0;
		
		let mEventInfos = new Array();
		let mMinDuration;
		let minDurationMinutes = Services.prefs.getIntPref('calendar.printing.day.mindurationminutes', 15);
		
		mMinDuration = Components.classes["@mozilla.org/calendar/duration;1"]
                                          .createInstance(Components.interfaces.calIDuration);
		aItems.forEach( function(item) {
			let normalizedItem = item.clone();
			normalizedItem = normalizedItem.wrappedJSObject || normalizedItem;
			
			let start = normalizedItem.startDate || normalizedItem.entryDate|| normalizedItem.dueDate;
			start = start.getInTimezone(defaultTimezone);
			normalizedItem.startDate = start;
			
			let end = item.endDate || item.dueDate || item.entryDate;
			end = end.getInTimezone(defaultTimezone);
			normalizedItem.endDate = end;
			
			// If the item starts before midnight, display as starting at midnight for today
			if(start.compare(dayStart) < 0) {
				normalizedItem.startDate = dayStart;
			} 
			// If the item end after midnight, display as ending at midnight today
			if(end.compare(dayEnd) > 0) {
				normalizedItem.endDate = dayEnd;
			}
			
			let secs = end.subtractDate(start).inSeconds;
			if(secs < (minDurationMinutes * 60)){
				let temp = mMinDuration.clone();
				temp.seconds = (minDurationMinutes * 60) - secs;
				end.addDuration(temp);
			}
			
			// We layout the item between the first and last shown hour
			if(start.compare(workingDayStart) < 0) {
				start = workingDayStart;
			} 
			if(end.compare(workingDayEnd) > 0) {
				end = workingDayEnd;
			}
		
			
			mEventInfos.push( {'event': normalizedItem, 'layoutStart': start, 'layoutEnd': end });
		});
	
		var blobs = new Array();
		var currentBlob = new Array();
		function sortByStart(aEventInfo, bEventInfo) {
			// If you pass in tasks without both entry and due dates, I will
			// kill you
			var startComparison = aEventInfo.layoutStart.compare(bEventInfo.layoutStart);
			if (startComparison != 0) {
				return startComparison;
			} else {
				// If the items start at the same time, return the longer one
				// first
				return bEventInfo.layoutEnd.compare(aEventInfo.layoutEnd);
			}
		}    
		mEventInfos.sort(sortByStart);
	
		// The end time of the last ending event in the entire blob
		var latestItemEnd;
	
		// This array keeps track of the last (latest ending) item in each of
		// the columns of the current blob. We could reconstruct this data at
		// any time by looking at the items in the blob, but that would hurt
		// perf.
		var colEndArray = new Array();
	
		/* Go through a 3 step process to try and place each item.
		 * Step 1: Look for an existing column with room for the item.
		 * Step 2: Look for a previously placed item that can be shrunk in
		 *         width to make room for the item.
         * Step 3: Give up and create a new column for the item.
         *
         * (The steps are explained in more detail as we come to them)
         */
		for (let i in mEventInfos) {
			var curItemInfo = {event: mEventInfos[i].event,
							layoutStart: mEventInfos[i].layoutStart,
							layoutEnd: mEventInfos[i].layoutEnd};
			if (!latestItemEnd) {
			latestItemEnd = curItemInfo.layoutEnd;
			}
			if (currentBlob.length && latestItemEnd &&
				curItemInfo.layoutStart.compare(latestItemEnd) != -1) {
				// We're done with this current blob because item starts
				// after the last event in the current blob ended.
				blobs.push({blob: currentBlob, totalCols: colEndArray.length});
	
				// Reset our variables
				currentBlob = new Array();
				colEndArray = new Array();
			}	
			// Place the item in its correct place in the blob
			var placedItem = false;
	
			// Step 1
			// Look for a possible column in the blob that has been left open. This
			// would happen if we already have multiple columns but some of
			// the cols have events before latestItemEnd.  For instance
			//       |      |      |
			//       |______|      |
			//       |ev1   |______|
			//       |      |ev2   |
			//       |______|      |
			//       |      |      |
			//       |OPEN! |      |<--Our item's start time might be here
			//       |      |______|
			//       |      |      |
			//
			// Remember that any time we're starting a new blob, colEndArray
			// will be empty, but that's ok.
			for (var ii = 0; ii < colEndArray.length; ++ii) {
				var colStart = colEndArray[ii].layoutStart;
				var colEnd = colEndArray[ii].layoutEnd;
				if (colEnd.compare(curItemInfo.layoutStart) != 1) {
					// Yay, we can jump into this column
					colEndArray[ii] = curItemInfo;
	
					// Check and see if there are any adjacent columns we can
					// jump into as well.
					var lastCol = Number(ii) + 1;
					while (lastCol < colEndArray.length) {
						var nextColStart = colEndArray[lastCol].layoutStart;
						var nextColEnd = colEndArray[lastCol].layoutEnd;
						// If the next column's item ends after we start, we
						// can't expand any further
						if (nextColEnd.compare(curItemInfo.layoutStart) == 1) {
							break;
						}
						colEndArray[lastCol] = curItemInfo;
						lastCol++;
					}
					// Now construct the info we need to push into the blob
					currentBlob.push({itemInfo: curItemInfo,
									startCol: ii,
									colSpan: lastCol - ii});
					// Update latestItemEnd
					if (latestItemEnd &&
						curItemInfo.layoutEnd.compare(latestItemEnd) == 1) {
						latestItemEnd = curItemInfo.layoutEnd;
					}
					placedItem = true;
					break; // Stop iterating through colEndArray
				}
			}
	
			if (placedItem) {
				// Go get the next item
				continue;
			}
	
			// Step 2
			// OK, all columns (if there are any) overlap us.  Look if the
			// last item in any of the last items in those columns is taking
			// up 2 or more cols. If so, shrink it and stick the item in the
			// created space. For instance
			//       |______|______|______|
			//       |ev1   |ev3   |ev4   |
			//       |      |      |      |
			//       |      |______|      |
			//       |      |      |______|
			//       |      |_____________|
			//       |      |ev2          |
			//       |______|             |<--If our item's start time is
			//       |      |_____________|   here, we can shrink ev2 and jump
			//       |      |      |      |   in column #3
			//
			for (var jj=1; jj<colEndArray.length; ++jj) {
				if (colEndArray[jj].event.hashId == colEndArray[jj-1].event.hashId) {
					// Good we found a item that spanned multiple columns.
					// Find it in the blob so we can modify its properties
					for (var kk in currentBlob) {
						if (currentBlob[kk].itemInfo.event.hashId == colEndArray[jj].event.hashId) {
							// Take all but the first spot that the item spanned
							var spanOfShrunkItem = currentBlob[kk].colSpan;
							currentBlob.push({itemInfo: curItemInfo,
											startCol: Number(currentBlob[kk].startCol) + 1,
											colSpan: spanOfShrunkItem - 1});

							// Update colEndArray
							for (var ll = jj; ll < jj + spanOfShrunkItem - 1; ll++) {
								colEndArray[ll] = curItemInfo;
							}
	
							// Modify the data on the old item
							currentBlob[kk] = {itemInfo: currentBlob[kk].itemInfo,
											startCol: currentBlob[kk].startCol,
											colSpan: 1};
							// Update latestItemEnd
							if (latestItemEnd &&
								curItemInfo.layoutEnd.compare(latestItemEnd) == 1) {
								latestItemEnd = curItemInfo.layoutEnd;
							}
							break; // Stop iterating through currentBlob
						}
					}
					placedItem = true;
					break; // Stop iterating through colEndArray
				}
			}
	
			if (placedItem) {
				// Go get the next item
				continue;
			}
	
			// Step 3
			// Guess what? We still haven't placed the item.  We need to
			// create a new column for it.
	
			// All the items in the last column, except for the one* that
			// conflicts with the item we're trying to place, need to have
			// their span extended by 1, since we're adding the new column
			//
			// * Note that there can only be one, because we sorted our
			//   events by start time, so this event must start later than
			//   the start of any possible conflicts.
			var lastColNum = colEndArray.length;
			for (var mm in currentBlob) {
				var mmStart = currentBlob[mm].itemInfo.layoutStart;
				var mmEnd = currentBlob[mm].itemInfo.layoutEnd;
				if (currentBlob[mm].startCol + currentBlob[mm].colSpan == lastColNum &&
					mmEnd.compare(curItemInfo.layoutStart) != 1) {
					currentBlob[mm] = {itemInfo: currentBlob[mm].itemInfo,
									startCol: currentBlob[mm].startCol,
									colSpan: currentBlob[mm].colSpan + 1};
				}
			}
			currentBlob.push({itemInfo: curItemInfo,
							startCol: colEndArray.length,
									colSpan: 1});
			colEndArray.push(curItemInfo);
	
			// Update latestItemEnd
			if (latestItemEnd && curItemInfo.layoutEnd.compare(latestItemEnd) == 1) {
				latestItemEnd = curItemInfo.layoutEnd;
			}
			// Go get the next item
		}
		// Add the last blob
		blobs.push({blob: currentBlob,
				totalCols: colEndArray.length});
		//return this.setupBoxStructure(blobs);
		return blobs;
	},
	
	//
	//  UTILS, might be common to other components ?
	//
	//
	isOffline: function isOffline(weekday) {
		const weekPrefix = "calendar.week.";
		let prefNames = ["d0sundaysoff", "d1mondaysoff", "d2tuesdaysoff",
						"d3wednesdaysoff", "d4thursdaysoff", "d5fridaysoff", "d6saturdaysoff"];
		let defaults = [true, false, false, false, false, false, true];
		
		return Services.prefs.getCharPref(weekPrefix+prefNames[weekday], defaults[weekday]);
	},
	
	
	getDateString: function dayPrinter_getDateString(aDate) {
		let dateFormatter = cal.getDateFormatter();
		let defaultTimezone;
		try {
			defaultTimezone = cal.dtz.defaultTimezone; 
		} catch(ex) {
			defaultTimezone = null;
		}
		let dateString;
		try {
			dateString = dateFormatter.dayName(aDate.getInTimezone(defaultTimezone).weekday) 
							+ " " 
							+ dateFormatter.formatDateShort(aDate.getInTimezone(defaultTimezone));
		} catch(ex) {
		    dateString = dateFormatter.dayName(aDate.weekday) + " " + dateFormatter.formatDateShort(aDate);
		}
		return dateString;
	},
	
	insertCalendarRules: function(document, calendar) {
        let sheet = document.getElementById("sheet");
        let color = calendar.getProperty("color") || "#A8C2E1";
        sheet.insertedCalendarRules = sheet.insertedCalendarRules || {};

        if (!(calendar.id in sheet.insertedCalendarRules)) {
            sheet.insertedCalendarRules[calendar.id] = true;
            let formattedId = cal.view.formatStringForCSSRule(calendar.id);
            let ruleAdd = ' .calendar-color-box[calendar-id="' + formattedId + '"] { ' +
                          " background-color: " + color + "; " +
                          " color: " + cal.view.getContrastingTextColor(color) + "; }\n";
            sheet.textContent += ruleAdd;
        }
    }
};