(function (exports) {

  // Import some things we need.
  var { MsgHdrToMimeMessage } = ChromeUtils.import("resource:///modules/gloda/MimeMessage.jsm");

  var tracker;

  class Tracker {
    constructor(extension) {
      this.windowTracker = new Set();
    }

    trackWindow(window) {
      this.windowTracker.add(window);
    }

    untrackWindow(window) {
      this.windowTracker.delete(window);
    }

    get trackedWindows() {
      return Array.from(this.windowTracker);
    }

  }

  class allowHtmlTemp extends ExtensionCommon.ExtensionAPI {
    // Alternative to defining a constructor here in order to init the class, is
    // to use the onStartup event. However, this causes the API to be instantiated
    // directly after the add-on has been loaded, not when the API is first used.
    constructor(extension) {
      // The only parameter is extension, but it could change in the future.
      // super() will add the extension as a member of this.
      super(extension);
      tracker = new Tracker(this.extension);
    }

    // The API implementation.
    getAPI(context) {
      
      function getMessageWindow(tabId) {
        // Get about:message from the tabId.
        let { nativeTab } = context.extension.tabManager.get(tabId);
        if (nativeTab instanceof Ci.nsIDOMWindow) {
          return nativeTab.messageBrowser.contentWindow
        } else if (nativeTab.mode && nativeTab.mode.name == "mail3PaneTab") {
          return nativeTab.chromeBrowser.contentWindow.messageBrowser.contentWindow
        } else if (nativeTab.mode && nativeTab.mode.name == "mailMessageTab") {
          return nativeTab.chromeBrowser.contentWindow;
        }
        return null;
      }

      return {
        allowHtmlTemp: {

          disconnectTab(tabId) {
            let messageWindow = getMessageWindow(tabId);
            if (!messageWindow) {
              return;
            }
            // Remove preference observers, to decouple this message tab from the
            // menu item.
            messageWindow.preferenceObserver.cleanUp();
            tracker.trackWindow(messageWindow);
          },

          reconnectTab(tabId) {
            let messageWindow = getMessageWindow(tabId);
            if (!messageWindow) {
              return;
            }
            // Connect the tab back to the prefs.
            messageWindow.preferenceObserver.init();
          },

          reloadTab(tabId) {
            let messageWindow = getMessageWindow(tabId);
            if (!messageWindow) {
              return;
            }
            // Connect the tab back to the prefs.
            messageWindow.ReloadMessage();
          },

          async checkMailForHtmlpart(messageId, optionsDebug) {
            if(optionsDebug)
              console.debug("AHT: run checkMailForHtmlpart ----------------");
            let ahtMsgHdr = context.extension.messageManager.get(messageId);
  
            // First check MsgHdr without decrypting to prevent an additional passphrase dialog in case of PGP/MIME
            let aMimeMsg = await new Promise(resolve => {
              MsgHdrToMimeMessage(
                ahtMsgHdr,
                null,
                (aMsgHdr, aMimeMsg) => resolve(aMimeMsg),
                true,
                {
                  examineEncryptedParts: false
                }
              );
            })
  
            // multipart/encrypted enables the button for encrypted PGP/MIME messages
            // in this case we don't check for HTML, because the check seems not to be possible for PGP/MIME
            if (aMimeMsg.prettyString().search("multipart/encrypted") != -1) {
              if(optionsDebug)
                console.debug("AHT: message is PGP/MIME multipart/encrypted");
              return true;
            } else {
              // search for 'Body: text/html' in MIME parts,
              // it seems this is only working if messages are downloaded for offline reading?
              let aMimeMsg = await new Promise(resolve => {
                MsgHdrToMimeMessage(
                  ahtMsgHdr,
                  null,
                  (aMsgHdr, aMimeMsg) => resolve(aMimeMsg),
                  true,
                  {
                    examineEncryptedParts: true
                  }
                );
              })
  
              if(optionsDebug) {
                console.debug("AHT: Check for html part ----------------");
                console.debug("AHT: Body: text/html " + aMimeMsg.prettyString().search("Body: text/html"));
                console.debug("AHT: text/html " + aMimeMsg.prettyString().search("text/html"));
                console.debug("AHT: Body: plain/html " + aMimeMsg.prettyString().search("Body: plain/html"));
                console.debug("AHT: plain/html " + aMimeMsg.prettyString().search("plain/html"));
                console.debug("AHT: multipart/alternative " + aMimeMsg.prettyString().search("multipart/alternative"));
                console.debug("AHT: multipart/signed " + aMimeMsg.prettyString().search("multipart/signed"));
                console.debug("AHT: multipart/encrypted " + aMimeMsg.prettyString().search("multipart/encrypted"));
              }
  
              // 'Body: text/html' is found, enable ahtButtons
              if (aMimeMsg.prettyString().search("Body: text/html") != -1) {
                if(optionsDebug)
                  console.debug("AHT: message contains HTML body part");
                return true;
              }
              // no 'Body: text/html', disable ahtButtons
              else {
                if(optionsDebug)
                  console.debug("AHT: no HTML body part");
                return false;
              }
            }
          }
  
        }
      };
    }

    onShutdown(isAppShutdown) {
      if (isAppShutdown) {
        return; // the application gets unloaded anyway
      }

      for(let messageWindow of tracker.trackedWindows) {
        if (!messageWindow) {
          continue;
        }

        // Connect the tab back to the prefs.
        messageWindow.preferenceObserver.init();
      }

      // Flush all caches
      Services.obs.notifyObservers(null, "startupcache-invalidate");
    }
  }

  exports.allowHtmlTemp = allowHtmlTemp;
})(this);
