// Since we disconnect all tabs from the policy prefs, we need to manually reload
// them on pref updates. If the pref change is a temporary add-on-inflicted change,
// we reload only a specific tab.
var ahtStatus = {
  reloadAllTabs: true
};

messenger.messageContentPolicy.onChanged.addListener(async (newValue) => {
  // If this policy change is the second part of an add-on inflicted temporary
  // policy change (returning to default), ignore it and reset the ahtStatus.
  if (ahtStatus.ignorePolicyChange) {
    ahtStatus = {
      reloadAllTabs: true
    }
    return;
  }

  let tabs = await messenger.tabs.query({});
  for (let tab of tabs) {
    if (!["messageDisplay", "mail"].includes(tab.type)) {
      continue;
    }
    if (ahtStatus.reloadAllTabs || ahtStatus.reloadTab == tab.id) {
      await messenger.allowHtmlTemp.reloadTab(tab.id);
    }
  }
  await setButtonIconAndLabel();
});

// An option change could cause the button policy to be identical to the current
// policy and the button needs to be disabled.
messenger.storage.onChanged.addListener(async (changes) => {
  await optionsInit();
  if (changes.buttonHtmlMode) {
    // Reload all already open tabs, to trigger the single entry point for all
    // UI updates of the AHT addon: a message display.
    let tabs = await messenger.tabs.query({});
    for (let tab of tabs) {
      if (!["messageDisplay", "mail"].includes(tab.type)) {
        continue;
      }
      await messenger.allowHtmlTemp.reloadTab(tab.id);
    }
  }
  await setButtonIconAndLabel();
  await setCommandKey();
});

// Add eventListener for onClicked on the message header button.
messenger.messageDisplayAction.onClicked.addListener(async (tab, info) => {
  let buttonPolicy = await getButtonPolicy();
  let currentPolicy = await getCurrentPolicy();

  // The following policy switch is not a change of the default, but a temporary
  // switch, so we have to return to the current policy once the tab is loaded.
  ahtStatus = {
    returnToPolicy: currentPolicy,
    reloadTab: tab.id,
  }
  // The following policy change will trigger a message reload (by our own listener),
  // which will return the contentPolicy to the former value.
  await messenger.messageContentPolicy.update({ msgBodyAs: buttonPolicy });
});

// Disconnect all message tabs from the menu/prefs and disable/enable buttons.
messenger.messageDisplay.onMessageDisplayed.addListener(async (tab, message) => {
  await messenger.allowHtmlTemp.disconnectTab(tab.id);

  // This is a reload caused by an add-on inflicted policy change, return to the
  // user defined value without reloading the tab again.
  if (ahtStatus.returnToPolicy) {
    // The following policy update should not trigger a reload.
    let msgBodyAs = ahtStatus.returnToPolicy;
    ahtStatus = {
      ignorePolicyChange: true
    }
    await messenger.messageContentPolicy.update({ msgBodyAs });
    
    // The button was used, so it no longer has an effect, disable it.
    disableButtons(tab.id);
  } else if (await isCurrentPolicy()) {
    // The current policy is identical to the button policy, so it has no effect,
    // disable it.
    disableButtons(tab.id);
  } else {
    // Update buttons for the newly loaded message.
    updateActionButtonForNewMessage(tab, message);
  }
});


(async () => {
  await optionsMigrate();
  await optionsInit();

  // Reload all already open tabs, to trigger the single entry point for all
  // UI updates of the AHT addon: a message display.
  let tabs = await messenger.tabs.query({});
  for (let tab of tabs) {
    if (!["messageDisplay", "mail"].includes(tab.type)) {
      continue;
    }
    await messenger.allowHtmlTemp.reloadTab(tab.id);
  }

})();

async function getButtonPolicy() {
  const modeMap = {
    "buttonMode_plaintext": "plaintext",
    "buttonMode_sanitized": "sanitized",
    "buttonMode_html": "original",
  }
  return modeMap[await getOption("buttonHtmlMode")];
}

async function getCurrentPolicy() {
  let { msgBodyAs: currentPolicy } = await messenger.messageContentPolicy.getCurrent();
  return currentPolicy;
}

async function isCurrentPolicy() {
  let buttonPolicy = await getButtonPolicy();
  let currentPolicy = await getCurrentPolicy();
  return buttonPolicy == currentPolicy;
}
